// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements k3d::selection_button, which provides a MVC UI for boolean data sources
		\author Tim Shead (tshead@k-3d.com)
*/

#include "selection_button.h"
#include "interactive.h"
#include "utility.h"

#include <k3dsdk/i18n.h>
#include <k3dsdk/istate_recorder.h>
#include <k3dsdk/state_change_set.h>

#include <gtkmm/button.h>

#include <sstream>

namespace libk3dngui
{

namespace selection_button
{

/////////////////////////////////////////////////////////////////////////////
// control

control::control(k3d::icommand_node& Parent, const std::string& Name, std::auto_ptr<idata_proxy> Data) :
	base(false, 0),
	ui_component(Name, &Parent),
	m_data(Data),
	m_select_all_button(new Gtk::Button(_("Select All"))),
	m_deselect_all_button(new Gtk::Button(_("Deselect All"))),
	m_select_null_button(new Gtk::Button(_("Clear")))
{
	set_name("k3d-selection-button");

	m_select_all_button->signal_clicked().connect(sigc::mem_fun(*this, &control::on_select_all));
	m_deselect_all_button->signal_clicked().connect(sigc::mem_fun(*this, &control::on_deselect_all));
	m_select_null_button->signal_clicked().connect(sigc::mem_fun(*this, &control::on_select_null));

	pack_start(*manage(m_select_all_button));
	pack_start(*manage(m_deselect_all_button));
	pack_start(*manage(m_select_null_button));
	
	update();

	if(m_data.get())
		m_data->changed_signal().connect(sigc::mem_fun(*this, &control::update));
}

bool control::execute_command(const std::string& Command, const std::string& Arguments)
{
	if(Command == "select_all")
	{
		interactive::activate(*m_select_all_button);
		return true;
	}
	else if(Command == "deselect_all")
	{
		interactive::activate(*m_deselect_all_button);
		return true;
	}
	else if(Command == "select_null")
	{
		interactive::activate(*m_select_null_button);
		return true;
	}
	
	return ui_component::execute_command(Command, Arguments);
}

void control::on_select_all()
{
	// Record the command for posterity (tutorials) ...
	record_command("select_all");

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->start_recording(k3d::create_state_change_set());

	// Update everything with the new value ...
	m_data->set_value(k3d::mesh_selection::select_all());

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), _("Select All"));
}

void control::on_deselect_all()
{
	// Record the command for posterity (tutorials) ...
	record_command("deselect_all");

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->start_recording(k3d::create_state_change_set());

	// Update everything with the new value ...
	m_data->set_value(k3d::mesh_selection::deselect_all());

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), _("Deselect All"));
}

void control::on_select_null()
{
	// Record the command for posterity (tutorials) ...
	record_command("select_null");

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->start_recording(k3d::create_state_change_set());

	// Update everything with the new value ...
	m_data->set_value(k3d::mesh_selection::select_null());

	// Turn this into an undo/redo -able event ...
	if(m_data->state_recorder)
		m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), _("Clear Selection"));
}

void control::update()
{
	if(!m_data.get())
		return;

	const k3d::mesh_selection selection = m_data->value();
	
	m_select_all_button->set_sensitive(selection != k3d::mesh_selection::select_all());
	m_deselect_all_button->set_sensitive(selection != k3d::mesh_selection::deselect_all());
	m_select_null_button->set_sensitive(selection != k3d::mesh_selection::select_null());
	
	std::ostringstream buffer;
	if(selection == k3d::mesh_selection::select_all())
	{
		buffer << _("All components are selected");
	}
	else if(selection == k3d::mesh_selection::deselect_all())
	{
		buffer << _("All components are deselected");
	}
	else if(selection == k3d::mesh_selection::select_null())
	{
		buffer << _("Using upstream component selection");
	}
	else
	{
		buffer << selection.points.size() << " " << selection.edges.size() << " " << selection.faces.size();
	}
	
	tooltips().set_tip(*m_select_all_button, buffer.str());
	tooltips().set_tip(*m_deselect_all_button, buffer.str());
	tooltips().set_tip(*m_select_null_button, buffer.str());
}

} // namespace selection_button

} // namespace libk3dngui


