// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "messages.h"
#include "scripting.h"

#include <k3dsdk/create_plugins.h>
#include <k3dsdk/fstream.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/log.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/scripting.h>
#include <k3dsdk/string_cast.h>
#include <k3dsdk/result.h>

#include <boost/filesystem/operations.hpp>
#include <boost/format.hpp>

#include <gtkmm/main.h>
#include <gdk/gdkkeysyms.h>

#include <iostream>
#include <stack>

namespace libk3dngui
{

namespace detail
{

std::stack<k3d::iscript_engine*> script_engine_stack;

/// Halts any running scripts if the user hits "escape"
int script_escape_handler(Gtk::Widget* Widget, GdkEventKey* Event)
{
	if(Event->type != GDK_KEY_RELEASE)
		return false;
		
	if(Event->keyval != GDK_Escape)
		return false;

	return_val_if_fail(script_engine_stack.size(), false);
	
	std::vector<std::string> buttons;
	buttons.push_back(_("Yes"));
	buttons.push_back(_("No"));
	if(1 == query_message(_("Halt running script?"), _("Halt Script:"), 2, buttons))
		script_engine_stack.top()->halt();
		
	return true;
}

/// Executes a script using the given plugin factory to create the script engine
bool execute_script(std::istream& Script, const std::string& ScriptName, k3d::iscript_engine::context_t& Context, k3d::iplugin_factory& Language)
{
	// Sanity checks ...
	return_val_if_fail(ScriptName.size(), false);

	// Get the requested scripting engine ...
	k3d::iscript_engine* const engine = k3d::create_plugin<k3d::iscript_engine>(Language);
	return_val_if_fail(engine, false);

	// Intercept global key events ...
	script_engine_stack.push(engine);
	sigc::connection script_escape_handler_connection = Gtk::Main::signal_key_snooper().connect(sigc::ptr_fun(script_escape_handler));

	// Run that bad-boy ...
	const bool result = engine->execute(ScriptName, Script, Context);
	
	script_escape_handler_connection.disconnect();
	script_engine_stack.pop();

	// Done with the engine ...
	delete dynamic_cast<k3d::ideletable*>(engine);

	return result;
}

void execute_script(std::istream& Script, const std::string& ScriptName, k3d::iscript_engine::context_t& Context, bool& Recognized, bool& Executed)
{
	// Starting state ...
	Recognized = false;
	Executed = false;

	// See which language it's written in ...
	k3d::iplugin_factory* const language = k3d::recognize_script_language(Script);

	// If the language wasn't recognized, we're done ...
	Recognized = language ? true : false;
	if(!Recognized)
		return;

	// Execute that puppy ...
	Executed = execute_script(Script, ScriptName, Context, *language);
}

bool execute_script(std::istream& Script, const std::string& ScriptName, k3d::iscript_engine::context_t& Context, const k3d::uuid& Language)
{
	// Lookup the engine ...
	k3d::iplugin_factory* const language = k3d::plugin(Language);
	return_val_if_fail(language, false);

	// Make it happen ...
	return execute_script(Script, ScriptName, Context, *language);
}

} // namespace detail
		
bool execute_script(const boost::filesystem::path& Script, k3d::iscript_engine::context_t& Context)
{
	if(!boost::filesystem::exists(Script))
	{
		error_message(
			k3d::string_cast(boost::format(_("Requested script file %1% doesn't exist.")) % Script.native_file_string()),
			k3d::string_cast(boost::format(_("Play %1%:")) % Script.native_file_string()));
		return false;
	}

	k3d::filesystem::ifstream script(Script);
	return execute_script(script, Script.native_file_string(), Context);
}

bool execute_script(const std::string& Script, const std::string& ScriptName, k3d::iscript_engine::context_t& Context)
{
	if(Script.empty())
	{
		error_message(
			_("Empty script cannot be executed."),
			k3d::string_cast(boost::format(_("Play %1%:")) % ScriptName));
		return false;
	}

	std::istringstream script(Script);
	return execute_script(script, ScriptName, Context);
}

bool execute_script(std::istream& Script, const std::string& ScriptName, k3d::iscript_engine::context_t& Context)
{
	// Make it happen ...
	bool recognized = false;
	bool executed = false;

	detail::execute_script(Script, ScriptName, Context, recognized, executed);

	if(!recognized)
	{
		error_message(
			_("Could not determine scripting language.  K-3D supports multiple scripting languages, but the language for this script was "
			"not recognized. Most K-3D script engines use some type of \"magic token\" at the beginning of a script to recognize it, e.g. \"#k3dscript\" "
			"in the first 12 characters of a script for K-3D's built-in K3DScript engine.  If you are writing a K-3D script, check the documentation "
			"for the scripting language you're writing in to see how to make it recognizable."),
			k3d::string_cast(boost::format(_("Play %1%:")) % ScriptName));
		return false;
	}

	if(!executed)
	{
		error_message(
			_("Error executing script"),
			k3d::string_cast(boost::format(_("Play %1%:")) % ScriptName));
		return false;
	}

	return true;
}

bool execute_script(std::istream& Script, const std::string& ScriptName, k3d::iscript_engine::context_t& Context, const k3d::uuid& Language)
{
	if(!detail::execute_script(Script, ScriptName, Context, Language))
	{
		error_message(
			_("Error executing script"),
			k3d::string_cast(boost::format(_("Play %1%:")) % ScriptName));
		return false;
	}

	return true;
}

} // namespace libk3dngui

