// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements k3d::script_button, which provides a MVC UI for boolean data sources
		\author Tim Shead (tshead@k-3d.com)
*/

#include "file_selection.h"
#include "interactive.h"
#include "script_button.h"
#include "utility.h"
#include "widget_manip.h"

#include <k3dsdk/i18n.h>
#include <k3dsdk/istate_recorder.h>
#include <k3dsdk/options.h>
#include <k3dsdk/state_change_set.h>
#include <k3dsdk/system.h>

#include <gtkmm/tooltips.h>

#include <k3dsdk/fstream.h>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/format.hpp>

#include <sstream>

namespace libk3dngui
{

namespace script_button
{

/////////////////////////////////////////////////////////////////////////////
// control

control::control(k3d::icommand_node& Parent, const std::string& Name, std::auto_ptr<idata_proxy> Data) :
	base(false, 0),
	ui_component(Name, &Parent),
	m_load(_("Load")),
	m_save(_("Save")),
	m_edit(_("Edit Script")),
	m_data(Data)
{
	m_load.signal_clicked().connect(sigc::mem_fun(*this, &control::on_load));
	m_save.signal_clicked().connect(sigc::mem_fun(*this, &control::on_save));
	m_edit.signal_clicked().connect(sigc::mem_fun(*this, &control::on_edit));

	pack_start(m_load, Gtk::PACK_SHRINK);
	pack_start(m_save, Gtk::PACK_SHRINK);
	pack_start(m_edit, Gtk::PACK_EXPAND_WIDGET);

	tooltips().set_tip(m_load, _("Load a script from disk, replacing the current script"));
	tooltips().set_tip(m_save, _("Save the current script to disk"));
	tooltips().set_tip(m_edit, _("Open the current script for editing"));

	set_name("k3d-script-button");

	// Update the display and ensure we get a notification anytime the data source changes ...
	return_if_fail(m_data.get());

	update();
	m_data->changed_signal().connect(sigc::mem_fun(*this, &control::update));
}

void control::update()
{
	return_if_fail(m_data.get());
}

void control::on_load()
{
	return_if_fail(m_data.get());

	boost::filesystem::path input_path;
	if(!get_file_path(k3d::ipath_property::READ, k3d::options::path::scripts(), _("Load Script"), boost::filesystem::path(), input_path))
		return;

	std::stringstream buffer;
	k3d::filesystem::ifstream input_stream(input_path);
	input_stream >> buffer.rdbuf();
	input_stream.close();

	if(buffer.str() != m_data->value())
		{
			if(m_data->state_recorder)
				m_data->state_recorder->start_recording(k3d::create_state_change_set());

			m_data->set_value(buffer.str());

			if(m_data->state_recorder)
				m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), m_data->change_message);
		}
}

void control::on_save()
{
	return_if_fail(m_data.get());

	boost::filesystem::path output_path;
	if(!get_file_path(k3d::ipath_property::WRITE, k3d::options::path::scripts(), _("Save Script"), boost::filesystem::path(), output_path))
		return;

	k3d::filesystem::ofstream output_stream(output_path);
	output_stream << m_data->value();
}

void control::on_edit()
{
	return_if_fail(m_data.get());

	try
		{
			const boost::filesystem::path temp_path = k3d::system::generate_temp_file();
			return_if_fail(!temp_path.empty());

			const std::string original_value = m_data->value();

			k3d::filesystem::ofstream output_stream(temp_path);
			output_stream << original_value;
			output_stream.close();

			boost::format command_line(k3d::options::get_command(k3d::options::command::text_editor()));
			command_line % temp_path.native_file_string();
			k3d::system::run_process(command_line.str());

			std::stringstream buffer;
			k3d::filesystem::ifstream input_stream(temp_path);
			input_stream >> buffer.rdbuf();
			input_stream.close();

			if(buffer.str() != original_value)
				{
					if(m_data->state_recorder)
						m_data->state_recorder->start_recording(k3d::create_state_change_set());

					m_data->set_value(buffer.str());

					if(m_data->state_recorder)
						m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), m_data->change_message);
				}

			boost::filesystem::remove(temp_path);
		}
	catch(std::exception& e)
		{
			k3d::log() << error << "Caught exception: " << e.what() << std::endl;
		}
	catch(...)
		{
			k3d::log() << error << "Caught unknown exception" << std::endl;
		}
}

bool control::execute_command(const std::string& Command, const std::string& Arguments)
{
	return ui_component::execute_command(Command, Arguments);
}

} // namespace script_button

} // namespace libk3dngui


