// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include <gtkmm/cellrenderertext.h>
#include <gtkmm/combobox.h>
#include <gtkmm/dialog.h>
#include <gtkmm/label.h>
#include <gtkmm/liststore.h>
#include <gtkmm/stock.h>

#include "document_state.h"
#include "file_selection.h"
#include "icons.h"
#include "menus.h"
#include "messages.h"

#include <k3dsdk/basic_math.h>
#include <k3dsdk/create_plugins.h>
#include <k3dsdk/frames.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/ianimation_render_engine.h>
#include <k3dsdk/icamera.h>
#include <k3dsdk/igl.h>
#include <k3dsdk/ipreview_depict_engine.h>
#include <k3dsdk/ipreview_render_engine.h>
#include <k3dsdk/istill_depict_engine.h>
#include <k3dsdk/istill_render_engine.h>
#include <k3dsdk/nodes.h>
#include <k3dsdk/options.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/property.h>
#include <k3dsdk/result.h>
#include <k3dsdk/string_cast.h>
#include <k3dsdk/time_source.h>

#include <boost/format.hpp>

namespace libk3dngui
{

namespace detail
{

k3d::icamera* pick_camera(document_state& DocumentState, const k3d::nodes_t& RenderEngines, const k3d::factories_t& Factories, const std::string& Title, const std::string& Message)
{
	class camera_columns :
		public Gtk::TreeModel::ColumnRecord
	{
	public:
		camera_columns()
		{
			add(object);
			add(factory);
			add(label);
			add(icon);
		}

		Gtk::TreeModelColumn<k3d::inode*> object;
		Gtk::TreeModelColumn<k3d::iplugin_factory*> factory;
		Gtk::TreeModelColumn<Glib::ustring> label;
		Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > icon;
	};

	camera_columns columns;
	Glib::RefPtr<Gtk::ListStore> model = Gtk::ListStore::create(columns);

	for(k3d::nodes_t::const_iterator object = RenderEngines.begin(); object != RenderEngines.end(); ++object)
	{
		Gtk::TreeModel::Row row = *model->append();
		row[columns.object] = *object;
		row[columns.factory] = 0;
		row[columns.icon] = quiet_load_icon((*object)->factory().name(), Gtk::ICON_SIZE_MENU);
		row[columns.label] = (*object)->name();
	}

	for(k3d::factories_t::const_iterator factory = Factories.begin(); factory != Factories.end(); ++factory)
	{
		std::string markup;
		if(k3d::iplugin_factory::EXPERIMENTAL == (*factory)->quality())
		{
			markup = k3d::string_cast(boost::format(_("<span color=\"blue\">Create %1% (Experimental)</span>")) % (*factory)->name());
		}
		else if(k3d::iplugin_factory::DEPRECATED == (*factory)->quality())
		{
			markup = k3d::string_cast(boost::format(_("<span color=\"red\" strikethrough=\"true\">Create %1%</span><span color=\"red\"> (Deprecated)</span>")) % (*factory)->name());
		}
		else
		{
			markup = k3d::string_cast(boost::format(_("Create %1%")) % (*factory)->name());
		}

		Gtk::TreeModel::Row row = *model->append();
		row[columns.object] = 0;
		row[columns.factory] = *factory;
		row[columns.icon] = quiet_load_icon((*factory)->name(), Gtk::ICON_SIZE_MENU);
		row[columns.label] = markup;
	}

	Gtk::ComboBox combo(model);

	combo.pack_start(columns.icon, false);

	Gtk::CellRendererText* const label_renderer = new Gtk::CellRendererText();
	combo.pack_start(*manage(label_renderer));
	combo.add_attribute(label_renderer->property_markup(), columns.label);

	combo.set_active(0);

	Gtk::Dialog dialog(Title, true);
	dialog.set_border_width(5);
	dialog.get_vbox()->pack_start(*Gtk::manage(new Gtk::Label(Message)), Gtk::PACK_SHRINK, 5);
	dialog.get_vbox()->pack_start(combo, Gtk::PACK_SHRINK, 5);
	dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
	dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	dialog.show_all();

	if(Gtk::RESPONSE_OK == dialog.run())
	{

		return_val_if_fail(combo.get_active() != model->children().end(), 0);

		if(k3d::inode* const object = combo.get_active()->get_value(columns.object))
			return dynamic_cast<k3d::icamera*>(object);

		if(k3d::iplugin_factory* const factory = combo.get_active()->get_value(columns.factory))
			return dynamic_cast<k3d::icamera*>(DocumentState.create_node(factory));
	}

	return 0;
}

template<typename interface_t>
interface_t* pick_render_engine(document_state& DocumentState, const k3d::nodes_t& RenderEngines, const k3d::factories_t& Factories, const std::string& Title, const std::string& Message)
{
	class render_engine_columns :
		public Gtk::TreeModel::ColumnRecord
	{
	public:
		render_engine_columns()
		{
			add(object);
			add(factory);
			add(label);
			add(icon);
		}

		Gtk::TreeModelColumn<k3d::inode*> object;
		Gtk::TreeModelColumn<k3d::iplugin_factory*> factory;
		Gtk::TreeModelColumn<Glib::ustring> label;
		Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > icon;
	};

	render_engine_columns columns;
	Glib::RefPtr<Gtk::ListStore> model = Gtk::ListStore::create(columns);

	for(k3d::nodes_t::const_iterator object = RenderEngines.begin(); object != RenderEngines.end(); ++object)
		{
			Gtk::TreeModel::Row row = *model->append();
			row[columns.object] = *object;
			row[columns.factory] = 0;
			row[columns.icon] = quiet_load_icon((*object)->factory().name(), Gtk::ICON_SIZE_MENU);
			row[columns.label] = (*object)->name();
		}

	for(k3d::factories_t::const_iterator factory = Factories.begin(); factory != Factories.end(); ++factory)
		{
			std::string markup;
			if(k3d::iplugin_factory::EXPERIMENTAL == (*factory)->quality())
				{
					markup = k3d::string_cast(boost::format(_("<span color=\"blue\">Create %1% (Experimental)</span>")) % (*factory)->name());
				}
			else if(k3d::iplugin_factory::DEPRECATED == (*factory)->quality())
				{
					markup = k3d::string_cast(boost::format(_("<span color=\"red\" strikethrough=\"true\">Create %1%</span><span color=\"red\"> (Deprecated)</span>")) % (*factory)->name());
				}
			else
				{
					markup = k3d::string_cast(boost::format(_("Create %1%")) % (*factory)->name());
				}

			Gtk::TreeModel::Row row = *model->append();
			row[columns.object] = 0;
			row[columns.factory] = *factory;
			row[columns.icon] = quiet_load_icon((*factory)->name(), Gtk::ICON_SIZE_MENU);
			row[columns.label] = markup;
		}

	Gtk::ComboBox combo(model);

	combo.pack_start(columns.icon, false);

	Gtk::CellRendererText* const label_renderer = new Gtk::CellRendererText();
	combo.pack_start(*manage(label_renderer));
	combo.add_attribute(label_renderer->property_markup(), columns.label);

	combo.set_active(0);

	Gtk::Dialog dialog(Title, true);
	dialog.set_border_width(5);
	dialog.get_vbox()->pack_start(*Gtk::manage(new Gtk::Label(Message)), Gtk::PACK_SHRINK, 5);
	dialog.get_vbox()->pack_start(combo, Gtk::PACK_SHRINK, 5);
	dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
	dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	dialog.show_all();

	if(Gtk::RESPONSE_OK == dialog.run())
		{

			return_val_if_fail(combo.get_active() != model->children().end(), 0);

			if(k3d::inode* const object = combo.get_active()->get_value(columns.object))
				return dynamic_cast<interface_t*>(object);

			if(k3d::iplugin_factory* const factory = combo.get_active()->get_value(columns.factory))
				{
					k3d::inode* const node = k3d::create_plugin<k3d::inode>(*factory, DocumentState.document(), k3d::unique_name(DocumentState.document().nodes(), factory->name()));
					DocumentState.view_node_properties_signal().emit(node);
					return dynamic_cast<interface_t*>(node);
				}
		}

	return 0;
}

} // namespace detail

k3d::icamera* default_camera(document_state& DocumentState)
{
	const k3d::nodes_t render_engines = k3d::find_nodes<k3d::icamera>(DocumentState.document().nodes());
	return render_engines.size() == 1 ? dynamic_cast<k3d::icamera*>(*render_engines.begin()) : 0;
}

k3d::gl::irender_engine* default_gl_render_engine(document_state& DocumentState)
{
	const k3d::nodes_t render_engines = k3d::find_nodes<k3d::gl::irender_engine>(DocumentState.document().nodes());
	return render_engines.size() == 1 ? dynamic_cast<k3d::gl::irender_engine*>(*render_engines.begin()) : 0;
}

k3d::ipreview_render_engine* default_preview_render_engine(document_state& DocumentState)
{
	const k3d::nodes_t render_engines = k3d::find_nodes<k3d::ipreview_render_engine>(DocumentState.document().nodes());
	return render_engines.size() == 1 ? dynamic_cast<k3d::ipreview_render_engine*>(*render_engines.begin()) : 0;
}

k3d::istill_render_engine* default_still_render_engine(document_state& DocumentState)
{
	const k3d::nodes_t render_engines = k3d::find_nodes<k3d::istill_render_engine>(DocumentState.document().nodes());
	return render_engines.size() == 1 ? dynamic_cast<k3d::istill_render_engine*>(*render_engines.begin()) : 0;
}

k3d::ianimation_render_engine* default_animation_render_engine(document_state& DocumentState)
{
	const k3d::nodes_t render_engines = k3d::find_nodes<k3d::ianimation_render_engine>(DocumentState.document().nodes());
	return render_engines.size() == 1 ? dynamic_cast<k3d::ianimation_render_engine*>(*render_engines.begin()) : 0;
}

k3d::icamera* pick_camera(document_state& DocumentState)
{
	const k3d::nodes_t render_engines = k3d::find_nodes<k3d::icamera>(DocumentState.document().nodes());
	const k3d::factories_t factories = k3d::plugins<k3d::icamera>();

	return detail::pick_camera(DocumentState, render_engines, factories, _("Pick Camera:"), _("Choose a camera"));
}

k3d::gl::irender_engine* pick_gl_render_engine(document_state& DocumentState)
{
	const k3d::nodes_t render_engines = k3d::find_nodes<k3d::gl::irender_engine>(DocumentState.document().nodes());
	const k3d::factories_t factories = k3d::plugins<k3d::gl::irender_engine>();

	return detail::pick_render_engine<k3d::gl::irender_engine>(DocumentState, render_engines, factories, _("Pick OpenGL Render Engine:"), _("Choose an OpenGL render engine to be used for drawing"));
}

k3d::ipreview_render_engine* pick_preview_render_engine(document_state& DocumentState)
{
	const k3d::nodes_t render_engines = k3d::find_nodes<k3d::ipreview_render_engine>(DocumentState.document().nodes());
	const k3d::factories_t factories = k3d::plugins<k3d::ipreview_render_engine>();

	return detail::pick_render_engine<k3d::ipreview_render_engine>(DocumentState, render_engines, factories, _("Pick Preview Render Engine:"), _("Choose a render engine to be used for preview image rendering"));
}

k3d::istill_render_engine* pick_still_render_engine(document_state& DocumentState)
{
	const k3d::nodes_t render_engines = k3d::find_nodes<k3d::istill_render_engine>(DocumentState.document().nodes());
	const k3d::factories_t factories = k3d::plugins<k3d::istill_render_engine>();

	return detail::pick_render_engine<k3d::istill_render_engine>(DocumentState, render_engines, factories, _("Pick Still Render Engine:"), _("Choose a render engine to be used for still image rendering"));
}

k3d::ianimation_render_engine* pick_animation_render_engine(document_state& DocumentState)
{
	const k3d::nodes_t render_engines = k3d::find_nodes<k3d::ianimation_render_engine>(DocumentState.document().nodes());
	const k3d::factories_t factories = k3d::plugins<k3d::ianimation_render_engine>();

	return detail::pick_render_engine<k3d::ianimation_render_engine>(DocumentState, render_engines, factories, _("Pick Animation Render Engine:"), _("Choose a render engine to be used for animation rendering"));
}

bool render_preview(document_state& DocumentState, k3d::icamera*& Camera, k3d::ipreview_render_engine*& Engine)
{
	if(!Camera)
		Camera = pick_camera(DocumentState);
	if(!Camera)
		return false;

	if(!Engine)
		Engine = pick_preview_render_engine(DocumentState);
	if(!Engine)
		return false;

	assert_warning(Engine->render_preview(*Camera));
	return true;
}

void render_frame(k3d::icamera& Camera, k3d::istill_render_engine& Engine)
{
	boost::filesystem::path file;
	if(!libk3dngui::get_file_path(k3d::ipath_property::WRITE, k3d::options::path::render_frame(), "Render Frame:", boost::filesystem::path(), file))
		return;

	assert_warning(Engine.render_frame(Camera, file, true));
}

void render_animation(document_state& DocumentState, k3d::icamera& Camera, k3d::ianimation_render_engine& Engine)
{
	// Ensure that the document has animation capabilities, first ...
	k3d::iproperty* const start_time_property = k3d::get_start_time(DocumentState.document());
	k3d::iproperty* const end_time_property = k3d::get_end_time(DocumentState.document());
	k3d::iproperty* const frame_rate_property = k3d::get_frame_rate(DocumentState.document());
	return_if_fail(start_time_property && end_time_property && frame_rate_property);

	// Prompt the user for a base filename ...
	boost::filesystem::path file;
	if(!libk3dngui::get_file_path(k3d::ipath_property::WRITE, k3d::options::path::render_animation(), "Choose Animation Base Filepath:", boost::filesystem::path(), file))
		return;

	// Make sure the supplied filepath has enough digits to render the entire animation ...
	const double start_time = boost::any_cast<double>(k3d::get_value(DocumentState.document().dag(), *start_time_property));
	const double end_time = boost::any_cast<double>(k3d::get_value(DocumentState.document().dag(), *end_time_property));
	const double frame_rate = boost::any_cast<double>(k3d::get_value(DocumentState.document().dag(), *frame_rate_property));

	const long start_frame = static_cast<long>(k3d::round(frame_rate * start_time));
	const long end_frame = static_cast<long>(k3d::round(frame_rate * end_time));

	k3d::frames frames(file, start_frame, end_frame);
	if(frames.max_frame() < end_frame)
		{
			std::string message = "The Base Filepath doesn't contain enough digits to render the entire animation.\n"
				"Try a filepath of the form [ myanim0000.tif ] ... the placement of digits is flexible,\n"
				"and any prefix / postfix / file extension is optional, but the path must contain\n"
				"enough consecutive digits to enumerate all of the frames in the animation.";

			error_message(message, "Render Animation:");
			return;
		}

	// See if the user wants to view frames as they're completed ...
	std::vector<std::string> buttons;
	buttons.push_back("Yes");
	buttons.push_back("No");
	buttons.push_back("Cancel");

	const unsigned long result = query_message("Do you want to see rendered frames as they're completed?", "Render Animation:", 1, buttons);
	if(0 == result || 3 == result)
		return;

	const bool viewcompleted = (1 == result);

	assert_warning(Engine.render_animation(Camera, file, viewcompleted));
}

k3d::ipreview_depict_engine* pick_preview_depict_engine(document_state& DocumentState)
{
	const k3d::nodes_t render_engines = k3d::find_nodes<k3d::ipreview_depict_engine>(DocumentState.document().nodes());
	const k3d::factories_t factories = k3d::plugins<k3d::ipreview_depict_engine>();

	return detail::pick_render_engine<k3d::ipreview_depict_engine>(DocumentState, render_engines, factories, _("Pick Preview Render Engine:"), _("Choose a render engine to be used for preview image rendering"));
}

k3d::istill_depict_engine* pick_still_depict_engine(document_state& DocumentState)
{
	const k3d::nodes_t render_engines = k3d::find_nodes<k3d::istill_depict_engine>(DocumentState.document().nodes());
	const k3d::factories_t factories = k3d::plugins<k3d::istill_depict_engine>();

	return detail::pick_render_engine<k3d::istill_depict_engine>(DocumentState, render_engines, factories, _("Pick Still Render Engine:"), _("Choose a render engine to be used for still image rendering"));
}

void depict_preview(k3d::ipreview_depict_engine& Engine)
{
	assert_warning(Engine.depict_preview());
}

void depict_frame(k3d::istill_depict_engine& Engine)
{
	boost::filesystem::path file;
	if(!libk3dngui::get_file_path(k3d::ipath_property::WRITE, k3d::options::path::render_frame(), "Render Frame:", boost::filesystem::path(), file))
		return;

	assert_warning(Engine.depict_frame(file, true));
}

} // namespace libk3dngui

