// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the k3d::enumeration_chooser classes, which provide a model-view-controller UI for enumerated (fixed set of choices) types
		\author Tim Shead (tshead@k-3d.com)
*/

#include "enumeration_chooser.h"
#include "interactive.h"
#include "options.h"
#include "utility.h"

#include <k3dsdk/command_tree.h>
#include <k3dsdk/istate_recorder.h>
#include <k3dsdk/state_change_set.h>

#include <gtkmm/entry.h>

namespace libk3dngui
{

namespace enumeration_chooser
{

/////////////////////////////////////////////////////////////////////////////
// control

control::control(k3d::icommand_node& Parent, const std::string& Name, std::auto_ptr<idata_proxy> Data, const k3d::ienumeration_property::enumeration_values_t& Choices) :
	ui_component(Name, &Parent),
	m_data(Data)
{
	if(m_data.get())
		m_data->changed_signal().connect(sigc::mem_fun(*this, &control::data_changed));

	m_model = Gtk::ListStore::create(m_columns);
	for(k3d::ienumeration_property::enumeration_values_t::const_iterator choice = Choices.begin(); choice != Choices.end(); ++choice)
		{
			Gtk::TreeModel::Row row = *m_model->append();
			row[m_columns.label] = choice->label;
			row[m_columns.value] = choice->value;
			row[m_columns.description] = choice->description;
		}
	set_model(m_model);

	Gtk::CellRendererText* const cell_renderer = new Gtk::CellRendererText();
	pack_start(*manage(cell_renderer), true);
	add_attribute(cell_renderer->property_text(), m_columns.label);

	data_changed();

	signal_changed().connect(sigc::mem_fun(*this, &control::on_list_changed));
}

bool control::execute_command(const std::string& Command, const std::string& Arguments)
{
	/** \todo Improve the tutorial interactivity of this thing */
	if(Command == "value")
		{
			const double speed = options::tutorial_speed();
			return_val_if_fail(speed, false);

			interactive::show(*this);
			interactive::move_pointer(*this);

			popup();

			const Gtk::TreeModel::Children children = m_model->children();
			for(Gtk::TreeModel::Children::const_iterator child = children.begin(); child != children.end(); ++child)
				{
					const std::string child_value = (*child)[m_columns.value];

					if(Arguments == child_value)
						{
							set_active(child);

							non_blocking_sleep(static_cast<unsigned long>(500 / speed));
							popdown();

							return true;
						}
				}

			k3d::log() << error << "Enumeration value [" << Arguments << "] does not match any allowed values" << std::endl;
			return false;
		}

	return ui_component::execute_command(Command, Arguments);
}

void control::data_changed()
{
	return_if_fail(m_data.get());

	const std::string value = m_data->value();

	const Gtk::TreeModel::Children children = m_model->children();
	for(Gtk::TreeModel::Children::const_iterator child = children.begin(); child != children.end(); ++child)
		{
			const std::string child_value = (*child)[m_columns.value];

			if(value == child_value)
				{
					set_active(child);
					return;
				}
		}

	k3d::log() << error << "Current enumeration value [" << value << "] does not match any allowed values" << std::endl;
}

void control::on_list_changed()
{
	set_new_value();
}

void control::set_new_value()
{
	Gtk::TreeModel::iterator active = get_active();
	return_if_fail(active != m_model->children().end());

	Gtk::TreeModel::Row row = *active;
	const std::string value = row[m_columns.value];

	k3d::command_tree().command_signal().emit(*this, k3d::icommand_node::INTERACTIVE, "value", value);

	return_if_fail(m_data.get());

	if(value == m_data->value())
		return;

	k3d::istate_recorder* const state_recorder = m_data->state_recorder;
	const Glib::ustring change_message = m_data->change_message;

	if(state_recorder)
		state_recorder->start_recording(k3d::create_state_change_set());

	m_data->set_value(value);

	if(state_recorder)
		state_recorder->commit_change_set(state_recorder->stop_recording(), change_message + " " + value);
}

} // namespace enumeration_chooser

} // namespace libk3dngui


