// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements k3d::check_menu_item, which provides a MVC UI for boolean data sources
		\author Tim Shead (tshead@k-3d.com)
*/

#include "interactive.h"
#include "check_menu_item.h"
#include "utility.h"

#include <k3dsdk/istate_recorder.h>
#include <k3dsdk/state_change_set.h>

#include <gtkmm/menu.h>
#include <gtkmm/tooltips.h>

namespace libk3dngui
{

namespace check_menu_item
{

/////////////////////////////////////////////////////////////////////////////
// control

control::control(k3d::icommand_node& Parent, const std::string& Name, std::auto_ptr<idata_proxy> Data) :
	ui_component(Name, &Parent),
	m_data(Data)
{
	set_name("k3d-check-menu-item");
	attach();
}

control::control(k3d::icommand_node& Parent, const std::string& Name, std::auto_ptr<idata_proxy> Data, const Glib::ustring& label, bool mnemonic) :
	base(label, mnemonic),
	ui_component(Name, &Parent),
	m_data(Data)
{
	set_name("k3d-check-menu-item");
	attach();
}

void control::attach()
{
	// Update the display ...
	update();

	// We want to be notified if the data source changes ...
	if(m_data.get())
		m_data->changed_signal().connect(sigc::mem_fun(*this, &control::update));
}


bool control::execute_command(const std::string& Command, const std::string& Arguments)
{
	if(Command == "toggle")
	{
		if(!is_sensitive())
			return true;

		typedef std::vector<Gtk::MenuItem*> menu_items_t;
		menu_items_t menu_items;

		for(Gtk::Widget* ancestor = this; ancestor; )
			{
				Gtk::MenuItem* const menu_item = dynamic_cast<Gtk::MenuItem*>(ancestor);
				if(menu_item)
					menu_items.push_back(menu_item);

				Gtk::Menu* const menu = dynamic_cast<Gtk::Menu*>(ancestor);
				if(menu)
					ancestor = menu->get_attach_widget();
				else
					ancestor = ancestor->get_parent();
			}

		// Make our item visible ...
		interactive::show(*this);

		// Now close everything down ...
		for(menu_items_t::iterator menu_item = menu_items.begin(); menu_item != menu_items.end(); ++menu_item)
			(*menu_item)->deselect();

		// Ensure that we let go of the pointer and keyboard ...
		get_display()->pointer_ungrab(GDK_CURRENT_TIME);
		get_display()->keyboard_ungrab(GDK_CURRENT_TIME);

		set_active(!get_active());
		handle_pending_events();
		non_blocking_sleep(500);

		return true;
	}

	return ui_component::execute_command(Command, Arguments);
}

void control::update()
{
	if(!m_data.get())
		return;

	const bool new_value = m_data->value();
	if(new_value != get_active())
		set_active(new_value);
}

void control::on_toggled()
{
	if(m_data.get())
	{
		// Get the control value ...
		const bool new_value = get_active();

		// If the value hasn't changed, we're done ...
		if(new_value != m_data->value())
		{
			// Record the command for posterity (tutorials) ...
			record_command("toggle");

			// Turn this into an undo/redo -able event ...
			if(m_data->state_recorder)
				m_data->state_recorder->start_recording(k3d::create_state_change_set());

			// Update everything with the new value ...
			m_data->set_value(new_value);

			// Turn this into an undo/redo -able event ...
			if(m_data->state_recorder)
				m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), new_value ? m_data->change_message + " \"On\"" : m_data->change_message + " \"Off\"");
		}
	}

	base::on_toggled();
}

} // namespace check_menu_item

} // namespace libk3dngui


