// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements k3d::check_button, which provides a MVC UI for boolean data sources
		\author Tim Shead (tshead@k-3d.com)
*/

#include "check_button.h"
#include "interactive.h"
#include "utility.h"

#include <k3dsdk/istate_recorder.h>
#include <k3dsdk/state_change_set.h>

#include <gtkmm/tooltips.h>

namespace libk3dngui
{

namespace check_button
{

/////////////////////////////////////////////////////////////////////////////
// control

control::control(k3d::icommand_node& Parent, const std::string& Name, std::auto_ptr<idata_proxy> Data) :
	ui_component(Name, &Parent),
	m_data(Data)
{
	set_name("k3d-check-button");
	attach();
}

control::control(k3d::icommand_node& Parent, const std::string& Name, std::auto_ptr<idata_proxy> Data, const Glib::ustring& label, bool mnemonic) :
	base(label, mnemonic),
	ui_component(Name, &Parent),
	m_data(Data)
{
	set_name("k3d-check-button");
	attach();
}

bool control::execute_command(const std::string& Command, const std::string& Arguments)
{
	if(Command == "value")
	{
		const bool new_value = Arguments == "true" ? true : false;
		if(new_value != get_active())
			interactive::activate(*this);

		return true;
	}

	return ui_component::execute_command(Command, Arguments);
}

void control::update()
{
	if(!m_data.get())
		return;

	const bool new_value = m_data->value();
	if(new_value != get_active())
		set_active(new_value);
}

void control::attach()
{
	// Update the display ...
	update();

	// We want to be notified if the data source changes ...
	if(m_data.get())
		m_data->changed_signal().connect(sigc::mem_fun(*this, &control::update));
}

void control::on_toggled()
{
	if(m_data.get())
	{
		// If the value hasn't changed, we're done ...
		const bool new_value = get_active();
		if(new_value != m_data->value())
		{
			// Record the command for posterity (tutorials) ...
			record_command("value", new_value ? "true" : "false");

			// Turn this into an undo/redo -able event ...
			if(m_data->state_recorder)
				m_data->state_recorder->start_recording(k3d::create_state_change_set());

			// Update everything with the new value ...
			m_data->set_value(new_value);

			// Turn this into an undo/redo -able event ...
			if(m_data->state_recorder)
				m_data->state_recorder->commit_change_set(m_data->state_recorder->stop_recording(), new_value ? m_data->change_message + " \"On\"" : m_data->change_message + " \"Off\"");
		}
	}

	base::on_toggled();
}

} // namespace check_button

} // namespace libk3dngui


