// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
  \author Timothy M. Shead (tshead@k-3d.com)
  \author Romain Behar (romainbehar@yahoo.com)
  */

#include <k3dsdk/i18n.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/transformable.h>
#include <k3dsdk/gl.h>

namespace libk3dselection
{

/////////////////////////////////////////////////////////////////////////////
// select_cube

class select_cube :
	public k3d::gl::drawable<k3d::transformable<k3d::mesh_modifier<k3d::persistent<k3d::node> > > >
{
	typedef k3d::gl::drawable<k3d::transformable<k3d::mesh_modifier<k3d::persistent<k3d::node> > > > base;

	public:
	select_cube(k3d::idocument& Document) :
		base(Document),
		m_x1(init_owner(*this) + init_name("x1") + init_label(_("X1")) + init_description(_("X coordinate of cube's min corner")) + init_value(-10.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_y1(init_owner(*this) + init_name("y1") + init_label(_("Y1")) + init_description(_("Y coordinate of cube's min corner")) + init_value(-10.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_z1(init_owner(*this) + init_name("z1") + init_label(_("Z1")) + init_description(_("Z coordinate of cube's min corner")) + init_value(-10.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_x2(init_owner(*this) + init_name("x2") + init_label(_("X2")) + init_description(_("X coordinate of cube's max corner")) + init_value(10.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_y2(init_owner(*this) + init_name("y2") + init_label(_("Y2")) + init_description(_("Y coordinate of cube's max corner")) + init_value(10.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_z2(init_owner(*this) + init_name("z2") + init_label(_("Z2")) + init_description(_("Z coordinate of cube's max corner")) + init_value(10.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance)))
	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &select_cube::on_reset_geometry));

		m_x1.changed_signal().connect(sigc::mem_fun(*this, &select_cube::on_reshape_geometry));
		m_y1.changed_signal().connect(sigc::mem_fun(*this, &select_cube::on_reshape_geometry));
		m_z1.changed_signal().connect(sigc::mem_fun(*this, &select_cube::on_reshape_geometry));
		m_x2.changed_signal().connect(sigc::mem_fun(*this, &select_cube::on_reshape_geometry));
		m_y2.changed_signal().connect(sigc::mem_fun(*this, &select_cube::on_reshape_geometry));
		m_z2.changed_signal().connect(sigc::mem_fun(*this, &select_cube::on_reshape_geometry));

		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &select_cube::on_create_geometry));
	}

	void on_gl_draw(const k3d::gl::render_state& State)
	{
		glDisable(GL_LIGHTING);
		glColor3d(1, 0, 0);

		const k3d::bounding_box3 cube(m_x1.value(), m_x2.value(), m_y1.value(), m_y2.value(), m_z1.value(), m_z2.value());
		if(cube.empty())
			return;

		k3d::gl::draw_bounding_box(cube);
	}

	void on_gl_select(const k3d::gl::render_state& State, const k3d::gl::select_state& SelectState)
	{
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		const k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return;

		k3d::mesh* const output = const_cast<k3d::mesh*>(m_output_mesh.internal_value());
		if(!output)
			return;

		reshape_geometry(*input, *output);

		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		reshape_geometry(*input, *output);

		return output;
	}

	struct select_volume
	{
		select_volume(const k3d::bounding_box3& Cube) :
			cube(Cube)
		{
		}

		template<typename T>
		void operator()(T& Component)
		{
			Component.selection_weight = 0;
			Component.visible_selection = false;
		}

		void operator()(k3d::point& Component)
		{
			const bool contained = cube.contains(Component.position);
			Component.selection_weight = contained;
			Component.visible_selection = contained;
		}

		const k3d::bounding_box3 cube;
	};

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		const k3d::bounding_box3 cube(
				m_x1.value(),
				m_x2.value(),
				m_y1.value(),
				m_y2.value(),
				m_z1.value(),
				m_z2.value());

		k3d::for_each_component(Target, select_volume(cube));
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<select_cube>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink> > > factory(
					k3d::uuid(0xd4d45c53, 0x7dd84306, 0x909462d9, 0xa45b7f4f),
					"SelectCube",
					_("Selects portions of the input mesh that intersect a cube"),
					"Selection",
					k3d::iplugin_factory::STABLE);

		return factory;
	}

	private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_x1;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_y1;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_z1;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_x2;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_y2;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_z2;
};

/////////////////////////////////////////////////////////////////////////////
// select_cube_factory

k3d::iplugin_factory& select_cube_factory()
{
	return select_cube::get_factory();
}

} // namespace libk3dselection

