// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the RenderManScript K-3D object, which can insert scripted data into RenderMan output
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/algebra.h>
#include <k3dsdk/create_plugins.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/iapplication.h>
#include <k3dsdk/iscript_engine.h>
#include <k3dsdk/itime_sink.h>
#include <k3dsdk/itransform_sink.h>
#include <k3dsdk/itransform_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/scripting.h>
#include <k3dsdk/user_property_changed_signal.h>

#define DEFAULT_SCRIPT "#python\n\nimport k3d\n\nk3d.OutputTransform = k3d.InputTransform\n\n"

namespace libk3dscripting
{

/////////////////////////////////////////////////////////////////////////////
// transform_modifier_script

class transform_modifier_script :
	public k3d::persistent<k3d::node>,
	public k3d::itransform_source,
	public k3d::itransform_sink
{
	typedef k3d::persistent<k3d::node>  base;

public:
	transform_modifier_script(k3d::idocument& Document) :
		base(Document),
		m_input(init_owner(*this) + init_name("input_matrix") + init_label(_("Input matrix")) + init_description(_("Input transformation matrix")) + init_value(k3d::identity3D())),
		m_script(init_owner(*this) + init_name("script") + init_label(_("Script")) + init_description(_("Script that modifies input matrix")) + init_value<std::string>(DEFAULT_SCRIPT)),
		m_output(init_owner(*this) + init_name("output_matrix") + init_label(_("Output matrix")) + init_description(_("Output matrix generated by the script")) + init_slot(sigc::mem_fun(*this, &transform_modifier_script::output_value))),
		m_script_engine(0),
		m_user_property_changed_signal(*this)
	{
		m_input.changed_signal().connect(sigc::mem_fun(*this, &transform_modifier_script::reset_output_cache));
		m_script.changed_signal().connect(sigc::mem_fun(*this, &transform_modifier_script::reset_output_cache));
		m_user_property_changed_signal.connect(sigc::mem_fun(*this, &transform_modifier_script::reset_output_cache));
	}

	~transform_modifier_script()
	{
		delete dynamic_cast<k3d::ideletable*>(m_script_engine);
	}

	k3d::iproperty& transform_source_output()
	{
		return m_output;
	}

	k3d::iproperty& transform_sink_input()
	{
		return m_input;
	}

	void reset_output_cache()
	{
		m_output_cache.reset();
		m_output.changed_signal().emit();
	}
	
	k3d::matrix4 output_value()
	{
		if(!m_output_cache.get())
		{
			// Create a new output matrix, ready for modification by the script ...
			const k3d::matrix4 input = m_input.value();
		
			m_output_cache.reset(new k3d::matrix4(input));

			// Examine the script to determine the correct language ...
			std::istringstream script(m_script.value());
			k3d::iplugin_factory* const language = k3d::recognize_script_language(script);
			return_val_if_fail(language, *m_output_cache);

			// If the current script engine is for the wrong language, lose it ...
			if(m_script_engine && (m_script_engine->factory().class_id() != language->class_id()))
			{
				delete dynamic_cast<k3d::ideletable*>(m_script_engine);
				m_script_engine = 0;
			}

			// Create our script engine as-needed ...
			if(!m_script_engine)
				m_script_engine = k3d::create_plugin<k3d::iscript_engine>(language->class_id());

			// No script engine?  We're outta here ...
			return_val_if_fail(m_script_engine, *m_output_cache);

			k3d::iscript_engine::context_t context;
			context["Document"] = static_cast<k3d::iunknown*>(&document());
			context["Object"] = static_cast<k3d::iunknown*>(this);
			context["InputTransform"] = input;
			context["OutputTransform"] = *m_output_cache;
			return_val_if_fail(m_script_engine->execute(name() + "Script", script, context), *m_output_cache);

			return_val_if_fail(context["OutputTransform"].type() == typeid(k3d::matrix4), *m_output_cache);
			*m_output_cache = boost::any_cast<k3d::matrix4>(context["OutputTransform"]);
		}

		return *m_output_cache;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<transform_modifier_script>,
			k3d::interface_list<k3d::itransform_sink, k3d::interface_list<k3d::itransform_source> > > factory(
			k3d::uuid(0xacafcc85, 0xa0bf4d69, 0x99592c4f, 0x7cf9b35c),
			"TransformModifierScript",
			_("Transform modifier that uses a script to modify a transformation matrix"),
			"Scripting",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(k3d::matrix4, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_input;
	k3d_data(std::string, immutable_name, change_signal, with_undo, local_storage, no_constraint, script_property, with_serialization) m_script;
	k3d_data(k3d::matrix4, immutable_name, change_signal, no_undo, computed_storage, no_constraint, read_only_property, no_serialization) m_output;
	k3d::iscript_engine* m_script_engine;
	k3d::user_property_changed_signal m_user_property_changed_signal;

	std::auto_ptr<k3d::matrix4> m_output_cache;
};

/////////////////////////////////////////////////////////////////////////////
// transform_modifier_script_factory

k3d::iplugin_factory& transform_modifier_script_factory()
{
	return transform_modifier_script::get_factory();
}

} // namespace libk3dscripting


