// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the RenderManScript K-3D object, which can insert scripted data into RenderMan output
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/create_plugins.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/iapplication.h>
#include <k3dsdk/iscript_engine.h>
#include <k3dsdk/bitmap_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/scripting.h>
#include <k3dsdk/user_property_changed_signal.h>

#define DEFAULT_SCRIPT "#python\n\nimport k3d\n\nk3d.OutputBitmap.reset(64, 64)\nfor x in range(64):\n\tfor y in range(64):\n\t\tk3d.OutputBitmap.set_pixel(x, y, k3d.color(1, 0, 0))\n\n"

namespace libk3dscripting
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_source_script

class bitmap_source_script :
	public k3d::bitmap_source<k3d::persistent<k3d::node> >
{
	typedef k3d::bitmap_source<k3d::persistent<k3d::node> > base;

public:
	bitmap_source_script(k3d::idocument& Document) :
		base(Document),
		m_script(init_owner(*this) + init_name("script") + init_label(_("Script")) + init_description(_("Script that generates a bitmap")) + init_value<std::string>(DEFAULT_SCRIPT)),
		m_script_engine(0),
		m_user_property_changed_signal(*this)
	{
		m_script.changed_signal().connect(sigc::mem_fun(*this, &bitmap_source_script::on_reset_bitmap));
		m_user_property_changed_signal.connect(sigc::mem_fun(*this, &bitmap_source_script::on_reset_bitmap));
		
		m_output.need_data_signal().connect(sigc::mem_fun(*this, &bitmap_source_script::on_create_bitmap));
	}

	~bitmap_source_script()
	{
		delete dynamic_cast<k3d::ideletable*>(m_script_engine);
	}

	void on_reset_bitmap()
	{
		m_output.reset();
	}

	k3d::bitmap* on_create_bitmap()
	{
		// Create the bitmap, then give the script a crack at it ...
		std::auto_ptr<k3d::bitmap> bitmap(new k3d::bitmap());

		// Examine the script to determine the correct language ...
		std::istringstream script(m_script.value());
		k3d::iplugin_factory* const language = k3d::recognize_script_language(script);
		return_val_if_fail(language, bitmap.release());

		// If the current script engine is for the wrong language, lose it ...
		if(m_script_engine && (m_script_engine->factory().class_id() != language->class_id()))
		{
			delete dynamic_cast<k3d::ideletable*>(m_script_engine);
			m_script_engine = 0;
		}

		// Create our script engine as-needed ...
		if(!m_script_engine)
			m_script_engine = k3d::create_plugin<k3d::iscript_engine>(language->class_id());

		// No script engine?  We're outta here ...
		return_val_if_fail(m_script_engine, bitmap.release());

		k3d::iscript_engine::context_t context;
		context["Document"] = static_cast<k3d::iunknown*>(&document());
		context["Object"] = static_cast<k3d::iunknown*>(this);
		context["OutputBitmap"] = bitmap.get();
		return_val_if_fail(m_script_engine->execute(name() + "Script", script, context), bitmap.release());

		return bitmap.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory< k3d::document_plugin<bitmap_source_script>, k3d::interface_list<k3d::ibitmap_source> > factory(
			k3d::uuid(0x98f6e0b6, 0x8423400b, 0xa5ae9144, 0x50e1c3cd),
			"BitmapSourceScript",
			_("Bitmap source that uses a script to generate images"),
			"Scripting Bitmap",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(std::string, immutable_name, change_signal, with_undo, local_storage, no_constraint, script_property, with_serialization) m_script;
	k3d::iscript_engine* m_script_engine;
	k3d::user_property_changed_signal m_user_property_changed_signal;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_source_script_factory

k3d::iplugin_factory& bitmap_source_script_factory()
{
	return bitmap_source_script::get_factory();
}

} // namespace libk3dscripting


