// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the RenderManShader plugin, which encapsulates a single RenderMan shader instance
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/application.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/color.h>
#include <k3dsdk/fstream.h>
#include <k3dsdk/iplugin_factory.h>
#include <k3dsdk/ipreview_depict_engine.h>
#include <k3dsdk/ipreview_render_engine.h>
#include <k3dsdk/iprojection.h>
#include <k3dsdk/irenderman.h>
#include <k3dsdk/irender_farm.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/property.h>
#include <k3dsdk/property_group_collection.h>
#include <k3dsdk/renderman.h>
#include <k3dsdk/shaders.h>
#include <k3dsdk/string_modifiers.h>

using namespace k3d::xml;

#ifdef	WIN32
#ifdef	interface
#undef	interface
#endif	//interface
#endif	//WIN32

#ifdef	WIN32
#ifdef	interface
#undef	interface
#endif	//interface
#endif	//WIN32

namespace libk3drenderman
{

/////////////////////////////////////////////////////////////////////////////
// shader_implementation

/// Abstract base class that provides most of the implementation for our shader objects
class shader_implementation :
	public k3d::persistent<k3d::node>
{
	typedef k3d::persistent<k3d::node> base;

public:
	shader_implementation(k3d::idocument& Document, const k3d::sl::shader::type_t ShaderType) :
		base(Document),
		m_shader_name(init_owner(*this) + init_name("shader_name") + init_label(_("Shader Name")) + init_description(_("RenderMan Shader Name")) + init_value<std::string>("") + init_values(shader_values(ShaderType)))
	{
		m_shader_name.changed_signal().connect(sigc::mem_fun(*this, &shader_implementation::on_shader_changed));
	}

	~shader_implementation()
	{
		clear();
	}

	void load(element& Element, const k3d::ipersistent::load_context& Context)
	{
		// Initialize the shader name in advance and create our shader arguments, so they can load themselves automatically
		std::string shader_name;
		if(element* const xml_properties = find_element(Element, "properties"))
		{
			shader_name = xml_properties->safe_element(element("property", attribute("name", "shader_name"))).text;
		}
		else if(element* const xml_properties = find_element(Element, "variables"))
		{
			shader_name = xml_properties->safe_element(element("variable", attribute("name", "shader_name"))).text;
			if(shader_name.empty())
				shader_name = attribute_text(xml_properties->safe_element(element("variable", attribute("name", "shader_name"))), "value");
		}
		m_shader_name.set_value(shader_name);

		base::load(Element, Context);
	}

protected:
	const std::string shader_name()
	{
		return m_shader_name.value();
	}

	k3d::ri::parameter_list shader_arguments(const k3d::ri::render_state& State)
	{
		k3d::ri::parameter_list results;

		for(arguments_t::iterator argument = m_arguments.begin(); argument != m_arguments.end(); ++argument)
			{
				if(argument->storage.empty())
					{
						// Empty storage is a placeholder for shader argument types that we don't currently support
					}
				else if(argument->storage.type() == typeid(string_storage_t*))
					{
						string_storage_t& storage = *boost::any_cast<string_storage_t*>(argument->storage);
						results.push_back(k3d::ri::parameter(argument->name, k3d::ri::CONSTANT, static_cast<k3d::ri::string>(storage.value())));
					}
				else if(argument->storage.type() == typeid(scalar_storage_t*))
					{
						scalar_storage_t& storage = *boost::any_cast<scalar_storage_t*>(argument->storage);
						results.push_back(k3d::ri::parameter(argument->name, k3d::ri::CONSTANT, static_cast<k3d::ri::real>(storage.value())));
					}
				else if(argument->storage.type() == typeid(angle_storage_t*))
					{
						angle_storage_t& storage = *boost::any_cast<angle_storage_t*>(argument->storage);
						results.push_back(k3d::ri::parameter(argument->name, k3d::ri::CONSTANT, static_cast<k3d::ri::real>(storage.value())));
					}
				else if(argument->storage.type() == typeid(color_storage_t*))
					{
						color_storage_t& storage = *boost::any_cast<color_storage_t*>(argument->storage);
						results.push_back(k3d::ri::parameter(argument->name, k3d::ri::CONSTANT, static_cast<k3d::ri::color>(storage.value())));
					}
				else if(argument->storage.type() == typeid(texture_storage_t*))
					{
						texture_storage_t& storage = *boost::any_cast<texture_storage_t*>(argument->storage);
						if(!storage.value())
							continue;

						results.push_back(k3d::ri::parameter(argument->name, k3d::ri::CONSTANT, static_cast<k3d::ri::string>(storage.value()->renderman_texture_path(State).native_file_string())));
					}
				else if(argument->storage.type() == typeid(point_storage_t*))
					{
						point_storage_t& storage = *boost::any_cast<point_storage_t*>(argument->storage);
						results.push_back(k3d::ri::parameter(argument->name, k3d::ri::CONSTANT, static_cast<k3d::ri::point>(storage.value())));
					}
				else if(argument->storage.type() == typeid(vector_storage_t*))
					{
						vector_storage_t& storage = *boost::any_cast<vector_storage_t*>(argument->storage);
						results.push_back(k3d::ri::parameter(argument->name, k3d::ri::CONSTANT, static_cast<k3d::ri::vector>(storage.value())));
					}
				else if(argument->storage.type() == typeid(normal_storage_t*))
					{
						normal_storage_t& storage = *boost::any_cast<normal_storage_t*>(argument->storage);
						results.push_back(k3d::ri::parameter(argument->name, k3d::ri::CONSTANT, static_cast<k3d::ri::normal>(storage.value())));
					}
				else
					{
						k3d::log() << error << k3d_file_reference << ": unknown storage type for shader argument [" << argument->name << "] will be ignored" << std::endl;
						continue;
					}
			}

		return results;
	}

private:
	/// Helper functor for searching for shaders by name
	struct same_name
	{
		same_name(const std::string& Name) : name(Name) {}

		template<typename T>
		bool operator()(const T& LHS)
		{
			return LHS.name == name;
		}

		const std::string name;
	};

	void on_shader_changed()
	{
		setup_arguments();
	}

	void clear()
	{
		// Get rid of old property / persistence proxies
		base::clear();

		for(arguments_t::iterator argument = m_arguments.begin(); argument != m_arguments.end(); ++argument)
			{
				// Empty storage is a placeholder for shader argument types that we don't currently support
				if(argument->storage.empty())
					{
					}
				else if(argument->storage.type() == typeid(scalar_storage_t*))
					{
						unregister_property(*boost::any_cast<scalar_storage_t*>(argument->storage));
					}
				else if(argument->storage.type() == typeid(angle_storage_t*))
					{
						unregister_property(*boost::any_cast<angle_storage_t*>(argument->storage));
					}
				else if(argument->storage.type() == typeid(color_storage_t*))
					{
						unregister_property(*boost::any_cast<color_storage_t*>(argument->storage));
					}
				else if(argument->storage.type() == typeid(string_storage_t*))
					{
						unregister_property(*boost::any_cast<string_storage_t*>(argument->storage));
					}
				else if(argument->storage.type() == typeid(texture_storage_t*))
					{
						unregister_property(*boost::any_cast<texture_storage_t*>(argument->storage));
					}
				else if(argument->storage.type() == typeid(point_storage_t*))
					{
						unregister_property(*boost::any_cast<point_storage_t*>(argument->storage));
					}
				else if(argument->storage.type() == typeid(vector_storage_t*))
					{
						unregister_property(*boost::any_cast<vector_storage_t*>(argument->storage));
					}
				else if(argument->storage.type() == typeid(normal_storage_t*))
					{
						unregister_property(*boost::any_cast<normal_storage_t*>(argument->storage));
					}
				else
					{
						k3d::log() << error << k3d_file_reference << ": unknown argument storage type for [" << argument->name << "] will not be destroyed" << std::endl;
					}
			}

		for(arguments_t::iterator argument = m_arguments.begin(); argument != m_arguments.end(); ++argument)
			{
				// Empty storage is a placeholder for shader argument types that we don't currently support
				if(argument->storage.empty())
					{
					}
				else if(argument->storage.type() == typeid(scalar_storage_t*))
					{
						k3d::undoable_delete(boost::any_cast<scalar_storage_t*>(argument->storage), document());
					}
				else if(argument->storage.type() == typeid(angle_storage_t*))
					{
						k3d::undoable_delete(boost::any_cast<angle_storage_t*>(argument->storage), document());
					}
				else if(argument->storage.type() == typeid(color_storage_t*))
					{
						k3d::undoable_delete(boost::any_cast<color_storage_t*>(argument->storage), document());
					}
				else if(argument->storage.type() == typeid(string_storage_t*))
					{
						k3d::undoable_delete(boost::any_cast<string_storage_t*>(argument->storage), document());
					}
				else if(argument->storage.type() == typeid(texture_storage_t*))
					{
						k3d::undoable_delete(boost::any_cast<texture_storage_t*>(argument->storage), document());
					}
				else if(argument->storage.type() == typeid(point_storage_t*))
					{
						k3d::undoable_delete(boost::any_cast<point_storage_t*>(argument->storage), document());
					}
				else if(argument->storage.type() == typeid(vector_storage_t*))
					{
						k3d::undoable_delete(boost::any_cast<vector_storage_t*>(argument->storage), document());
					}
				else if(argument->storage.type() == typeid(normal_storage_t*))
					{
						k3d::undoable_delete(boost::any_cast<normal_storage_t*>(argument->storage), document());
					}
				else
					{
						k3d::log() << error << k3d_file_reference << ": unknown argument storage type for [" << argument->name << "] will not be destroyed" << std::endl;
					}
			}

		m_arguments.clear();
	}

	class pause_recording
	{
	public:
		pause_recording(k3d::idocument& Document) :
			m_document(Document)
		{
			if(m_document.state_recorder().current_change_set())
				m_change_set = m_document.state_recorder().stop_recording();
		}

		~pause_recording()
		{
			if(m_change_set.get())
				m_document.state_recorder().start_recording(m_change_set);
		}

	private:
		k3d::idocument& m_document;
		std::auto_ptr<k3d::state_change_set> m_change_set;
	};

	void setup_arguments()
	{
		// At the moment, we don't support undo-able changes to the argument-set, so temporarily disable any recording ...
		pause_recording pause(document());

		clear();

		enable_serialization(m_shader_name.name(), m_shader_name);

		// Try to find a shader description that matches the new shader name ...
		const std::string shader_name = m_shader_name.value();
		if(shader_name.empty())
			return;

		const k3d::sl::shaders_t::const_iterator shader = std::find_if(k3d::application().shaders().begin(), k3d::application().shaders().end(), same_name(shader_name));
		if(shader == k3d::application().shaders().end())
			{
				k3d::log() << error << "Unknown shader [" << shader_name << "]" << std::endl;
				return;
			}

		for(k3d::sl::shader::arguments_t::const_iterator argument = shader->arguments.begin(); argument != shader->arguments.end(); ++argument)
			{
				if(argument->output)
					continue;

				switch(argument->extended_type)
					{
						case k3d::sl::argument::EX_FLOAT:
						case k3d::sl::argument::EX_TIME:
						case k3d::sl::argument::EX_DISTANCE:
						case k3d::sl::argument::EX_AREA:
						case k3d::sl::argument::EX_VOLUME:
						case k3d::sl::argument::EX_MASS:
						case k3d::sl::argument::EX_FORCE:
						case k3d::sl::argument::EX_PRESSURE:
							{
								scalar_storage_t* const scalar_storage = new scalar_storage_t(init_owner(*this) + init_name(argument->name.c_str()) + init_label(argument->label.c_str()) + init_description(argument->description.c_str()) + init_value(k3d::from_string<double>(argument->default_value, 0.0)));

								m_arguments.push_back(argument_t(argument->name, boost::any(scalar_storage)));
								break;
							}
						case k3d::sl::argument::EX_ANGLE:
							{
								angle_storage_t* const angle_storage = new angle_storage_t(init_owner(*this) + init_name(argument->name.c_str()) + init_label(argument->label.c_str()) + init_description(argument->description.c_str()) + init_value(k3d::radians(k3d::from_string<double>(argument->default_value, 0.0))) + init_units(typeid(k3d::measurement::angle)));

								m_arguments.push_back(argument_t(argument->name, boost::any(angle_storage)));
								break;
							}
						case k3d::sl::argument::EX_COLOR:
							{
								color_storage_t* const color_storage = new color_storage_t(init_owner(*this) + init_name(argument->name.c_str()) + init_label(argument->label.c_str()) + init_description(argument->description.c_str()) + init_value(k3d::from_string<k3d::color>(argument->default_value, k3d::color(1, 1, 1))));

								m_arguments.push_back(argument_t(argument->name, boost::any(color_storage)));
								break;
							}
						case k3d::sl::argument::EX_STRING:
						case k3d::sl::argument::EX_SPACE:
							{
								string_storage_t* const string_storage = new string_storage_t(init_owner(*this) + init_name(argument->name.c_str()) + init_label(argument->label.c_str()) + init_description(argument->description.c_str()) + init_value(argument->default_value));

								m_arguments.push_back(argument_t(argument->name, boost::any(string_storage)));
								break;
							}
						case k3d::sl::argument::EX_TEXTURE:
							{
								texture_storage_t* const texture_storage = new texture_storage_t(init_owner(*this) + init_name(argument->name.c_str()) + init_label(argument->label.c_str()) + init_description(argument->description.c_str()) + init_value<k3d::ri::itexture*>(0));

								m_arguments.push_back(argument_t(argument->name, boost::any(texture_storage)));
								break;
							}
						case k3d::sl::argument::EX_POINT:
							{
								point_storage_t* const vector_storage = new point_storage_t(init_owner(*this) + init_name(argument->name.c_str()) + init_label(argument->label.c_str()) + init_description(argument->description.c_str()) + init_value(k3d::from_string<k3d::vector3>(argument->default_value, k3d::vector3(0, 0, 0))));

								m_arguments.push_back(argument_t(argument->name, boost::any(vector_storage)));
								break;
							}
						case k3d::sl::argument::EX_VECTOR:
							{
								vector_storage_t* const vector_storage = new vector_storage_t(init_owner(*this) + init_name(argument->name.c_str()) + init_label(argument->label.c_str()) + init_description(argument->description.c_str()) + init_value(k3d::from_string<k3d::vector3>(argument->default_value, k3d::vector3(0, 0, 0))));

								m_arguments.push_back(argument_t(argument->name, boost::any(vector_storage)));
								break;
							}
						case k3d::sl::argument::EX_NORMAL:
							{
								normal_storage_t* const vector_storage = new normal_storage_t(init_owner(*this) + init_name(argument->name.c_str()) + init_label(argument->label.c_str()) + init_description(argument->description.c_str()) + init_value(k3d::from_string<k3d::vector3>(argument->default_value, k3d::vector3(0, 0, 0))));

								m_arguments.push_back(argument_t(argument->name, boost::any(vector_storage)));
								break;
							}
						case k3d::sl::argument::EX_MATRIX:
							{
								// Empty storage is a placeholder for shader argument types that we don't currently support
								m_arguments.push_back(argument_t(argument->name, boost::any()));
								break;
							}
						default:
							{
								m_arguments.push_back(argument_t(argument->name, boost::any()));
								k3d::log() << error << k3d_file_reference << " unknown extended argument type for [" << argument->name << "] will not receive storage" << std::endl;
							}
					}
			}
	}

	/// Storage for the shader name
	k3d_data(std::string, immutable_name, change_signal, with_undo, local_storage, no_constraint, list_property, with_serialization) m_shader_name;

	/// Defines storage for a single shader argument instance
	struct argument_t
	{
		argument_t(const std::string& Name, const boost::any Storage) : name(Name), storage(Storage) {}
		const std::string name;
		const boost::any storage;
	};

	/// Defines storage for a collection of shader arguments
	typedef std::list<argument_t> arguments_t;

	/// Stores the current set of shader arguments
	arguments_t m_arguments;

	/// Defines storage for a scalar argument
	typedef k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) scalar_storage_t;
	/// Defines storage for an angle argument (in radians)
	typedef k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) angle_storage_t;
	/// Defines storage for a color argument
	typedef k3d_data(k3d::color, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) color_storage_t;
	/// Defines storage for a string argument
	typedef k3d_data(std::string, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) string_storage_t;
	/// Defines storage for a texture argument
	typedef k3d_data(k3d::ri::itexture*, immutable_name, change_signal, with_undo, node_storage, no_constraint, node_property, node_serialization) texture_storage_t;
	/// Defines storage for a point argument
	typedef k3d_data(k3d::vector3, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) point_storage_t;
	/// Defines storage for a vector argument
	typedef k3d_data(k3d::vector3, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) vector_storage_t;
	/// Defines storage for a normal argument
	typedef k3d_data(k3d::vector3, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) normal_storage_t;

	/// Returns lists of available shaders of the given type
	const k3d::ilist_property<std::string>::values_t& shader_values(const k3d::sl::shader::type_t Type)
	{
		static bool initialized = false;
		static k3d::ilist_property<std::string>::values_t deformation_shader_values;
		static k3d::ilist_property<std::string>::values_t imager_shader_values;
		static k3d::ilist_property<std::string>::values_t light_shader_values;
		static k3d::ilist_property<std::string>::values_t surface_shader_values;
		static k3d::ilist_property<std::string>::values_t transformation_shader_values;
		static k3d::ilist_property<std::string>::values_t volume_shader_values;
		static k3d::ilist_property<std::string>::values_t empty_shader_values;

		if(!initialized)
			{
				for(k3d::sl::shaders_t::const_iterator shader = k3d::application().shaders().begin(); shader != k3d::application().shaders().end(); ++shader)
					{
						switch(shader->type)
							{
								case k3d::sl::shader::DISPLACEMENT:
									deformation_shader_values.push_back(shader->name);
									break;

								case k3d::sl::shader::IMAGER:
									imager_shader_values.push_back(shader->name);
									break;

								case k3d::sl::shader::LIGHT:
									light_shader_values.push_back(shader->name);
									break;

								case k3d::sl::shader::SURFACE:
									surface_shader_values.push_back(shader->name);
									break;

								case k3d::sl::shader::TRANSFORMATION:
									transformation_shader_values.push_back(shader->name);
									break;

								case k3d::sl::shader::VOLUME:
									volume_shader_values.push_back(shader->name);
									break;
							}
					}

				deformation_shader_values.push_back(std::string());
				imager_shader_values.push_back(std::string());
				light_shader_values.push_back(std::string());
				surface_shader_values.push_back(std::string());
				transformation_shader_values.push_back(std::string());
				volume_shader_values.push_back(std::string());

				std::sort(deformation_shader_values.begin(), deformation_shader_values.end());
				std::sort(imager_shader_values.begin(), imager_shader_values.end());
				std::sort(light_shader_values.begin(), light_shader_values.end());
				std::sort(surface_shader_values.begin(), surface_shader_values.end());
				std::sort(transformation_shader_values.begin(), transformation_shader_values.end());
				std::sort(volume_shader_values.begin(), volume_shader_values.end());

				initialized = true;
			}

		switch(Type)
			{
				case k3d::sl::shader::DISPLACEMENT:
					return deformation_shader_values;

				case k3d::sl::shader::IMAGER:
					return imager_shader_values;

				case k3d::sl::shader::LIGHT:
					return light_shader_values;

				case k3d::sl::shader::SURFACE:
					return surface_shader_values;

				case k3d::sl::shader::TRANSFORMATION:
					return transformation_shader_values;

				case k3d::sl::shader::VOLUME:
					return volume_shader_values;
			}

		return empty_shader_values;
	}

};

/////////////////////////////////////////////////////////////////////////////
// displacement_shader_implementation

/// Implements a RenderMan displacement shader plugin
class displacement_shader_implementation :
	public shader_implementation,
	public k3d::ri::idisplacement_shader
{
	typedef shader_implementation base;

public:
	displacement_shader_implementation(k3d::idocument& Document) :
		base(Document, k3d::sl::shader::DISPLACEMENT)
	{
	}

	void setup_renderman_displacement_shader(const k3d::ri::render_state& State)
	{
		State.engine.RiDisplacementV(shader_name(), shader_arguments(State));
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<displacement_shader_implementation>,
				k3d::interface_list<k3d::ri::idisplacement_shader> > factory(
			k3d::classes::RenderManDisplacementShader(),
			"RenderManDisplacementShader",
			_("Encapsulates a RenderMan displacement shader instance"),
			"RenderMan",
			k3d::iplugin_factory::STABLE);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// imager_shader_implementation

/// Implements a RenderMan imager shader plugin
class imager_shader_implementation :
	public shader_implementation,
	public k3d::ri::iimager_shader
{
	typedef shader_implementation base;

public:
	imager_shader_implementation(k3d::idocument& Document) :
		base(Document, k3d::sl::shader::IMAGER)
	{
	}

	void setup_renderman_imager_shader(const k3d::ri::render_state& State)
	{
		State.engine.RiImagerV(shader_name(), shader_arguments(State));
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<imager_shader_implementation>,
				k3d::interface_list<k3d::ri::iimager_shader> > factory(
			k3d::classes::RenderManImagerShader(),
			"RenderManImagerShader",
			"Encapsulates a RenderMan imager shader instance",
			"RenderMan",
			k3d::iplugin_factory::STABLE);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// light_shader_implementation

/// Implements a RenderMan light shader plugin
class light_shader_implementation :
	public shader_implementation,
	public k3d::ri::ilight_shader
{
	typedef shader_implementation base;

public:
	light_shader_implementation(k3d::idocument& Document) :
		base(Document, k3d::sl::shader::LIGHT)
	{
	}

	void setup_renderman_light_shader(const k3d::ri::render_state& State)
	{
		State.engine.RiLightSourceV(shader_name(), shader_arguments(State));
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<light_shader_implementation>,
				k3d::interface_list<k3d::ri::ilight_shader> > factory(
			k3d::classes::RenderManLightShader(),
			"RenderManLightShader",
			"Encapsulates a RenderMan light shader instance",
			"RenderMan",
			k3d::iplugin_factory::STABLE);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// surface_shader_implementation

/// Implements a RenderMan surface shader plugin
class surface_shader_implementation :
	public shader_implementation,
	public k3d::ri::isurface_shader,
	public k3d::ipreview_depict_engine,
	public k3d::property_group_collection
{
	typedef shader_implementation base;

public:
	surface_shader_implementation(k3d::idocument& Document) :
		base(Document, k3d::sl::shader::SURFACE),
		m_left(init_owner(*this) + init_name("left") + init_label(_("Left")) + init_description(_("Left")) + init_value(-2.0/3.0) + init_precision(2) + init_step_increment(0.01) + init_units(typeid(k3d::measurement::scalar))),
		m_right(init_owner(*this) + init_name("right") + init_label(_("Right")) + init_description(_("Right")) + init_value(2.0/3.0) + init_precision(2) + init_step_increment(0.01) + init_units(typeid(k3d::measurement::scalar))),
		m_top(init_owner(*this) + init_name("top") + init_label(_("Top")) + init_description(_("Top")) + init_value(0.5) + init_precision(2) + init_step_increment(0.01) + init_units(typeid(k3d::measurement::scalar))),
		m_bottom(init_owner(*this) + init_name("bottom") + init_label(_("Bottom")) + init_description(_("Bottom")) + init_value(-0.5) + init_precision(2) + init_step_increment(0.01) + init_units(typeid(k3d::measurement::scalar))),
		m_near(init_owner(*this) + init_name("near") + init_label(_("Near")) + init_description(_("Near Plane Distance")) + init_value(1.0) + init_constraint(constraint::minimum(0.0)) + init_precision(1) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_far(init_owner(*this) + init_name("far") + init_label(_("Far")) + init_description(_("Far Plane Distance")) + init_value(1000.0) + init_constraint(constraint::minimum(0.0)) + init_precision(1) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_perspective_projection(m_left, m_right, m_top, m_bottom, m_near, m_far)
	{
		k3d::iproperty_group_collection::group preview_group("Preview");
		preview_group.properties.push_back(&static_cast<k3d::iproperty&>(m_left));
		preview_group.properties.push_back(&static_cast<k3d::iproperty&>(m_right));
		preview_group.properties.push_back(&static_cast<k3d::iproperty&>(m_top));
		preview_group.properties.push_back(&static_cast<k3d::iproperty&>(m_bottom));
		preview_group.properties.push_back(&static_cast<k3d::iproperty&>(m_near));
		preview_group.properties.push_back(&static_cast<k3d::iproperty&>(m_far));

		register_property_group(preview_group);
	}

	void setup_renderman_surface_shader(const k3d::ri::render_state& State)
	{
		State.engine.RiSurfaceV(shader_name(), shader_arguments(State));
	}

	bool depict_preview()
	{
		// Start a new render job
		k3d::irender_job& job = k3d::application().render_farm().create_job("k3d-shader-preview");

		// Add a single frame to the job
		k3d::irender_frame& frame = job.create_frame("frame");

		// Start our RIB file
		const std::string ribfilename("world.rib");
		const boost::filesystem::path ribfilepath = frame.add_input_file(ribfilename);

		// Open the RIB file stream
		boost::filesystem::ofstream ribfile(ribfilepath);

		// Setup RenderMan engine
		k3d::inode* renderman_engine = 0;
		const k3d::nodes_t render_engines = k3d::find_nodes<k3d::ipreview_render_engine>(document().nodes());
		for(k3d::nodes_t::const_iterator object = render_engines.begin(); object != render_engines.end(); ++object)
			{
				if(k3d::classes::RenderManEngine() == (*object)->factory().class_id())
					{
						renderman_engine = *object;
						break;
					}
			}

		return_val_if_fail(renderman_engine, false);

		// Get render engine name
		k3d::iproperty* const engine_property = k3d::get_property(*renderman_engine, "render_engine");
		return_val_if_fail(engine_property, false);
		const std::string render_engine = boost::any_cast<std::string>(engine_property->property_value());

		// Create the RenderMan render engine object
		k3d::ri::render_engine engine(ribfile);

		// Setup frame
		frame.add_render_operation("ri", render_engine, ribfilename, true);

		// Create the render state object
		k3d::ri::sample_times_t samples(1, 0.0);
		const k3d::matrix4 transform_matrix = k3d::translation3D(k3d::vector3(-20, 20, -20)) * k3d::rotation3D(k3d::vector3(35, 45, 0));
		k3d::ri::render_state state(frame, engine, m_perspective_projection, k3d::ri::FINAL_FRAME, samples, 0, transform_matrix);

		// Setup options
		k3d::ri::parameter_list searchpath_options;
		searchpath_options.push_back(k3d::ri::parameter("shader", k3d::ri::UNIFORM, k3d::application().shader_cache_path().native_file_string() + ":&"));
		engine.RiOptionV("searchpath", searchpath_options);

		engine.RiDisplayV("outputimage", k3d::ri::RI_FRAMEBUFFER(), k3d::ri::RI_RGB());
		engine.RiFormat(320, 240, 1);

		// Create RenderMan scene
		const double left = boost::any_cast<double>(k3d::get_value(document().dag(), m_perspective_projection.left()));
		const double right = boost::any_cast<double>(k3d::get_value(document().dag(), m_perspective_projection.right()));
		const double top = boost::any_cast<double>(k3d::get_value(document().dag(), m_perspective_projection.top()));
		const double bottom = boost::any_cast<double>(k3d::get_value(document().dag(), m_perspective_projection.bottom()));
		const double near = boost::any_cast<double>(k3d::get_value(document().dag(), m_perspective_projection.near()));
		const double far = boost::any_cast<double>(k3d::get_value(document().dag(), m_perspective_projection.far()));

		engine.RiProjectionV("perspective");
		engine.RiScreenWindow(left, right, bottom, top);
		engine.RiClipping(near, far);

		engine.RiRotate(0, 0, 0, 1);
		engine.RiRotate(-35, 1, 0, 0);
		engine.RiRotate(-90, 0, 1, 0);
		engine.RiTranslate(10, -7, 0);

		engine.RiWorldBegin();

		// Setup a light
		engine.RiTransformBegin();
		engine.RiTransform(k3d::ri::convert(k3d::translation3D(k3d::vector3(-20, 20, -25))));
		k3d::ri::parameter_list parameters;
		parameters.push_back(k3d::ri::parameter(std::string("intensity"), k3d::ri::CONSTANT, (k3d::ri::real)3000.0));
		parameters.push_back(k3d::ri::parameter(std::string("lightcolor"), k3d::ri::CONSTANT, (k3d::ri::color)k3d::color(1, 1, 1)));
		engine.RiLightSourceV("k3d_pointlight", parameters);
		engine.RiTransformEnd();

		// Setup shader
		setup_renderman_surface_shader(state);

		// Default object
		engine.RiCylinderV(5.0, -5.0, 5.0, 360.0);

		engine.RiWorldEnd();

		// Synchronize shader
		for(k3d::sl::shaders_t::const_iterator shader = k3d::application().shaders().begin(); shader != k3d::application().shaders().end(); ++shader)
			{
				// No match
				if(shader->name != shader_name())
					continue;

				// Compile it
				if(!k3d::compile_shader(boost::filesystem::path(shader->file_path, boost::filesystem::native), "ri", render_engine))
					k3d::log() << error << k3d::string_cast(boost::format(_("Error compiling shader %1%")) % shader->name) << std::endl;
			}

		// Run the job
		k3d::application().render_farm().start_job(job);

		return true;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<surface_shader_implementation>,
				k3d::interface_list<k3d::ri::isurface_shader> > factory(
			k3d::classes::RenderManSurfaceShader(),
			"RenderManSurfaceShader",
			"Encapsulates a RenderMan surface shader instance",
			"RenderMan",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

	k3d_data(double, immutable_name, change_signal, no_undo, local_storage, no_constraint, measurement_property, no_serialization) m_left;
	k3d_data(double, immutable_name, change_signal, no_undo, local_storage, no_constraint, measurement_property, no_serialization) m_right;
	k3d_data(double, immutable_name, change_signal, no_undo, local_storage, no_constraint, measurement_property, no_serialization) m_top;
	k3d_data(double, immutable_name, change_signal, no_undo, local_storage, no_constraint, measurement_property, no_serialization) m_bottom;
	k3d_data(double, immutable_name, change_signal, no_undo, local_storage, with_constraint, measurement_property, no_serialization) m_near;
	k3d_data(double, immutable_name, change_signal, no_undo, local_storage, with_constraint, measurement_property, no_serialization) m_far;

	class perspective_projection :
		public k3d::iperspective
	{
	public:
		perspective_projection(k3d::iproperty& Left, k3d::iproperty& Right, k3d::iproperty& Top, k3d::iproperty& Bottom, k3d::iproperty& Near, k3d::iproperty& Far) :
			m_left(Left),
			m_right(Right),
			m_top(Top),
			m_bottom(Bottom),
			m_near(Near),
			m_far(Far)
		{
		}

		k3d::iproperty& left()
		{
			return m_left;
		}

		k3d::iproperty& right()
		{
			return m_right;
		}

		k3d::iproperty& top()
		{
			return m_top;
		}

		k3d::iproperty& bottom()
		{
			return m_bottom;
		}

		k3d::iproperty& near()
		{
			return m_near;
		}

		k3d::iproperty& far()
		{
			return m_far;
		}

	private:
		k3d::iproperty& m_left;
		k3d::iproperty& m_right;
		k3d::iproperty& m_top;
		k3d::iproperty& m_bottom;
		k3d::iproperty& m_near;
		k3d::iproperty& m_far;
	};

	perspective_projection m_perspective_projection;
};

/////////////////////////////////////////////////////////////////////////////
// volume_shader_implementation

/// Implements a RenderMan volume shader plugin
class volume_shader_implementation :
	public shader_implementation,
	public k3d::ri::ivolume_shader
{
	typedef shader_implementation base;

public:
	volume_shader_implementation(k3d::idocument& Document) :
		base(Document, k3d::sl::shader::VOLUME)
	{
	}

	void setup_renderman_atmosphere_shader(const k3d::ri::render_state& State)
	{
		State.engine.RiAtmosphereV(shader_name(), shader_arguments(State));
	}

	void setup_renderman_interior_shader(const k3d::ri::render_state& State)
	{
		State.engine.RiInteriorV(shader_name(), shader_arguments(State));
	}

	void setup_renderman_exterior_shader(const k3d::ri::render_state& State)
	{
		State.engine.RiExteriorV(shader_name(), shader_arguments(State));
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<volume_shader_implementation>,
				k3d::interface_list<k3d::ri::ivolume_shader> > factory(
			k3d::classes::RenderManVolumeShader(),
			"RenderManVolumeShader",
			"Encapsulates a RenderMan volume shader instance",
			"RenderMan",
			k3d::iplugin_factory::STABLE);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// shader factories

k3d::iplugin_factory& displacement_shader_factory()
{
	return displacement_shader_implementation::get_factory();
}

k3d::iplugin_factory& imager_shader_factory()
{
	return imager_shader_implementation::get_factory();
}

k3d::iplugin_factory& light_shader_factory()
{
	return light_shader_implementation::get_factory();
}

k3d::iplugin_factory& surface_shader_factory()
{
	return surface_shader_implementation::get_factory();
}

k3d::iplugin_factory& volume_shader_factory()
{
	return volume_shader_implementation::get_factory();
}

} // namespace libk3drenderman


