// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_client.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>

#include <iterator>

namespace libk3dprimitives
{

/////////////////////////////////////////////////////////////////////////////
// poly_cylinder_implementation

class poly_cylinder_implementation :
	public k3d::material_client<k3d::mesh_source<k3d::persistent<k3d::node> > >
{
	typedef k3d::material_client<k3d::mesh_source<k3d::persistent<k3d::node> > > base;

public:
	poly_cylinder_implementation(k3d::idocument& Document) :
		base(Document),
		m_u_segments(init_owner(*this) + init_name("u_segments") + init_label(_("U segments")) + init_description(_("Columns")) + init_value(32) + init_constraint(constraint::minimum(3L)) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar))),
		m_v_segments(init_owner(*this) + init_name("v_segments") + init_label(_("V segments")) + init_description(_("Rows")) + init_value(5) + init_constraint(constraint::minimum(1L)) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar))),
		m_top(init_owner(*this) + init_name("top") + init_label(_("Top")) + init_description(_("Cap cylinder top")) + init_value(true)),
		m_bottom(init_owner(*this) + init_name("bottom") + init_label(_("Bottom")) + init_description(_("Cap cylinder bottom")) + init_value(true)),
		m_radius(init_owner(*this) + init_name("radius") + init_label(_("Radius")) + init_description(_("Cylinder radius")) + init_value(5.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_zmax(init_owner(*this) + init_name("zmax") + init_label(_("Z max")) + init_description(_("Z max (RenderMan convention)")) + init_value(5.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_zmin(init_owner(*this) + init_name("zmin") + init_label(_("Z min")) + init_description(_("Z min (RenderMan convention)")) + init_value(-5.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_u_power(init_owner(*this) + init_name("u_power") + init_label(_("U power")) + init_description(_("Radial power")) + init_value(1.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::scalar)))
	{
		m_material.changed_signal().connect(sigc::mem_fun(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_u_segments.changed_signal().connect(sigc::mem_fun(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_v_segments.changed_signal().connect(sigc::mem_fun(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_top.changed_signal().connect(sigc::mem_fun(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_bottom.changed_signal().connect(sigc::mem_fun(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_radius.changed_signal().connect(sigc::mem_fun(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_zmax.changed_signal().connect(sigc::mem_fun(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_zmin.changed_signal().connect(sigc::mem_fun(*this, &poly_cylinder_implementation::on_reset_geometry));
		m_u_power.changed_signal().connect(sigc::mem_fun(*this, &poly_cylinder_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &poly_cylinder_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		const double radius = m_radius.value();
		const double zmax = m_zmax.value();
		const double zmin = m_zmin.value();
		const double u_power = m_u_power.value();
		const double inv_u_power = u_power ? 1.0 / u_power : 1.0;
		const bool top = m_top.value();
		const bool bottom = m_bottom.value();
		k3d::imaterial* const material = m_material.value();

		std::auto_ptr<k3d::mesh> mesh(new k3d::mesh());

		mesh->polyhedra.push_back(new k3d::polyhedron());
		k3d::polyhedron& polyhedron = *mesh->polyhedra.back();

		// Create the cylinder body ...
		const k3d::grid_results_t grid = k3d::add_grid(*mesh, polyhedron, m_v_segments.value(), m_u_segments.value(), false, true, material);
		const boost::multi_array<k3d::point*, 2>& points = boost::get<0>(grid);
		const boost::multi_array<k3d::split_edge*, 3>& edges = boost::get<1>(grid);

		// Define the shape of the cylinder ...
		const unsigned long point_v_segments = points.shape()[0];
		const unsigned long point_u_segments = points.shape()[1];

		// Optionally cap the top of the cylinder ...
		if(top)
			{
				std::vector<k3d::split_edge*> new_edges;
				for(unsigned long u = 0; u != point_u_segments; ++u)
					new_edges.push_back(new k3d::split_edge(edges[0][(point_u_segments - u) % point_u_segments][0]->vertex));

				for(unsigned long u = 0; u != point_u_segments; ++u)
					{
						new_edges[u]->face_clockwise = new_edges[(u + 1) % point_u_segments];
						k3d::join_edges(*new_edges[u], *edges[0][(point_u_segments-1)-u][0]);
					}

				polyhedron.faces.push_back(new k3d::face(new_edges[0], material));
			}

		// Optionally cap the bottom of the cylinder ...
		if(bottom)
			{
				std::vector<k3d::split_edge*> new_edges;
				for(unsigned long u = 0; u != point_u_segments; ++u)
					new_edges.push_back(new k3d::split_edge(edges[point_v_segments-2][u][3]->vertex));

				for(unsigned long u = 0; u != point_u_segments; ++u)
					{
						new_edges[u]->face_clockwise = new_edges[(u + 1) % point_u_segments];
						k3d::join_edges(*new_edges[u], *edges[point_v_segments-2][u][2]);
					}

				polyhedron.faces.push_back(new k3d::face(new_edges[0], material));
			}

		// Shape the cylinder points
		for(unsigned long v = 0; v != point_v_segments; ++v)
			{
				const double height = static_cast<double>(v) / static_cast<double>(point_v_segments - 1);

				for(unsigned long u = 0; u != point_u_segments; ++u)
					{
						const double theta = k3d::pi_times_2() * static_cast<double>(u) / static_cast<double>(point_u_segments);

						double x = cos(theta);
						double y = -sin(theta);
						double z = k3d::mix(zmax, zmin, height);

						x = radius * k3d::sign(x) * std::pow(std::abs(x), inv_u_power);
						y = radius * k3d::sign(y) * std::pow(std::abs(y), inv_u_power);

						points[v][u]->position = k3d::vector3(x, y, z);
					}
			}

		return_val_if_fail(is_valid(polyhedron), 0);

		if(top && bottom)
			assert_warning(is_solid(polyhedron));

		return mesh.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<poly_cylinder_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
			k3d::uuid(0xd8c4d9fd, 0x42334a54, 0xa4b48185, 0xd8506489),
			"PolyCylinder",
			_("Generates a polygonal cylinder with optional endcaps"),
			"Polygons",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, with_constraint, measurement_property, with_serialization) m_u_segments;
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, with_constraint, measurement_property, with_serialization) m_v_segments;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_top;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_bottom;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_radius;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_zmax;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_zmin;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_u_power;
};

/////////////////////////////////////////////////////////////////////////////
// poly_cylinder_factory

k3d::iplugin_factory& poly_cylinder_factory()
{
	return poly_cylinder_implementation::get_factory();
}

} // namespace libk3dprimitives


