// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/axis.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/material_client.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/mesh_selection_sink.h>
#include <k3dsdk/module.h>

#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// set_material_implementation

class set_material_implementation :
	public k3d::material_client<k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > > >
{
	typedef k3d::material_client<k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > > > base;

public:
	set_material_implementation(k3d::idocument& Document) :
		base(Document)
	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &set_material_implementation::on_reset_geometry));
		m_mesh_selection.changed_signal().connect(sigc::mem_fun(*this, &set_material_implementation::on_reset_geometry));
		m_material.changed_signal().connect(sigc::mem_fun(*this, &set_material_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &set_material_implementation::on_create_geometry));
	}
	
	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}
	
	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input mesh and modify the copy ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		k3d::replace_selection(m_mesh_selection.value(), *output);
		reshape_geometry(*input, *output);

		return output;
	}

	class set_material
	{
	public:
		set_material(k3d::imaterial* const Material) :
			m_material(Material)
		{
		}
	
		template<typename T>
		void operator()(T& Object)
		{
			if(Object.selection_weight)
				Object.material = m_material;
		}

		void operator()(k3d::point&) { }
		void operator()(k3d::split_edge&) { }
		void operator()(k3d::linear_curve&) { }
		void operator()(k3d::cubic_curve&) { }
		void operator()(k3d::nucurve&) { }
		void operator()(k3d::polyhedron&) { }
		void operator()(k3d::mesh&) { }

	private:
		k3d::imaterial* const m_material;
	};

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		k3d::for_each_component(Target, set_material(m_material.value()));

	}
	
	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<set_material_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x7743a7f2, 0x995345b9, 0x8d6dacc7, 0x3fa37029),
				"SetMaterial",
				_("Sets geometry materials"),
				"Materials",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
};

/////////////////////////////////////////////////////////////////////////////
// set_material_factory

k3d::iplugin_factory& set_material_factory()
{
	return set_material_implementation::get_factory();
}

} // namespace libk3dmesh

