// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/mesh_selection_sink.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>

namespace libk3dmesh
{

namespace detail
{

/// Handles negative numbers the way I'd expect it to
long mymod(const long a, const long b)
{
	long result = a;

	while(result < 0)
		result += b;
	while(result >= b)
		result -= b;

	return result;
}

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// move_first_edge_implementation

class move_first_edge_implementation :
	public k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > >
{
	typedef k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > > base;

public:
	move_first_edge_implementation(k3d::idocument& Document) :
		base(Document),
		m_distance(init_owner(*this) + init_name("distance") + init_label(_("distance")) + init_description(_("Face distance")) + init_value(0UL) + init_precision(0) + init_step_increment(1) + init_units(typeid(k3d::measurement::scalar)))
	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &move_first_edge_implementation::on_reset_geometry));
		m_mesh_selection.changed_signal().connect(sigc::mem_fun(*this, &move_first_edge_implementation::on_reset_geometry));
		m_distance.changed_signal().connect(sigc::mem_fun(*this, &move_first_edge_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &move_first_edge_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		k3d::replace_selection(m_mesh_selection.value(), *output);

		const long distance = m_distance.value();

		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = output->polyhedra.begin(); polyhedron != output->polyhedra.end(); ++polyhedron)
			{
				for(k3d::polyhedron::faces_t::iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
					{
						if(!(*face)->selection_weight)
							continue;

						std::vector<k3d::split_edge*> edges;
						for(k3d::split_edge* edge = (*face)->first_edge; edge; edge = edge->face_clockwise)
							{
								edges.push_back(edge);
								if(edge->face_clockwise == (*face)->first_edge)
									break;
							}

						(*face)->first_edge = edges[detail::mymod(distance, edges.size())];
					}
			}

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<move_first_edge_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x81c766ed, 0x2cb943c7, 0x91e9e680, 0xeba9bc4b),
				"MoveFirstEdge",
				_("Changes the first edge for selected polygons"),
				"Polygons",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(long, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_distance;
};

/////////////////////////////////////////////////////////////////////////////
// move_first_edge_factory

k3d::iplugin_factory& move_first_edge_factory()
{
	return move_first_edge_implementation::get_factory();
}

} // namespace libk3dmesh


