// K-3D
// Copyright (c) 2005, Romain Behar
//
// Contact: romainbehar@yahoo.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/mesh_selection_sink.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/utility.h>

#include <list>
#include <set>

namespace libk3dmesh
{

namespace detail
{

typedef std::set<k3d::point*> point_set_t;

/// Removes used k3d::point* elements from PointSet
class remove_used_points
{
public:
	remove_used_points(point_set_t& PointSet) :
		point_set(PointSet)
	{
	}

	void operator()(k3d::point&) { }
	void operator()(k3d::face&) { }

	void operator()(k3d::split_edge& Edge)
	{
		if(!point_set.size())
			return;

		point_set_t::const_iterator point = point_set.find(Edge.vertex);
		if(point != point_set.end())
			point_set.erase(point);
	}

	void operator()(k3d::linear_curve& Curve)
	{
		if(!point_set.size())
			return;

		for(k3d::linear_curve::control_points_t::iterator curve_point = Curve.control_points.begin(); curve_point != Curve.control_points.end(); ++curve_point)
			{
				point_set_t::const_iterator point = point_set.find(*curve_point);
				if(point != point_set.end())
					point_set.erase(point);
			}
	}

	void operator()(k3d::linear_curve_group&) { }

	void operator()(k3d::cubic_curve& Curve)
	{
		if(!point_set.size())
			return;

		for(k3d::cubic_curve::control_points_t::iterator curve_point = Curve.control_points.begin(); curve_point != Curve.control_points.end(); ++curve_point)
			{
				point_set_t::const_iterator point = point_set.find(*curve_point);
				if(point != point_set.end())
					point_set.erase(point);
			}
	}

	void operator()(k3d::cubic_curve_group&) { }

	void operator()(k3d::nucurve& Curve)
	{
		if(!point_set.size())
			return;

		for(k3d::nucurve::control_points_t::iterator curve_point = Curve.control_points.begin(); curve_point != Curve.control_points.end(); ++curve_point)
			{
				point_set_t::const_iterator point = point_set.find(curve_point->position);
				if(point != point_set.end())
					point_set.erase(point);
			}
	}

	void operator()(k3d::nucurve_group&) { }

	void operator()(k3d::bilinear_patch& Patch)
	{
		if(!point_set.size())
			return;

		for(k3d::bilinear_patch::control_points_t::iterator control_point = Patch.control_points.begin(); control_point != Patch.control_points.end(); ++control_point)
			{
				point_set_t::const_iterator point = point_set.find(*control_point);
				if(point != point_set.end())
					point_set.erase(point);
			}
	}

	void operator()(k3d::bicubic_patch& Patch)
	{
		if(!point_set.size())
			return;

		for(k3d::bicubic_patch::control_points_t::iterator control_point = Patch.control_points.begin(); control_point != Patch.control_points.end(); ++control_point)
			{
				point_set_t::const_iterator point = point_set.find(*control_point);
				if(point != point_set.end())
					point_set.erase(point);
			}
	}

	void operator()(k3d::nupatch& Patch)
	{
		if(!point_set.size())
			return;

		for(k3d::nupatch::control_points_t::iterator control_point = Patch.control_points.begin(); control_point != Patch.control_points.end(); ++control_point)
			{
				point_set_t::const_iterator point = point_set.find(control_point->position);
				if(point != point_set.end())
					point_set.erase(point);
			}
	}

	void operator()(k3d::polyhedron&) { }
	void operator()(k3d::mesh&) { }

private:
	point_set_t& point_set;
};

/// Returns whether k3d::point* is in given set
struct is_in_point_set
{
	is_in_point_set(point_set_t& PointSet) :
		point_set(PointSet)
	{
	}

	bool operator()(k3d::point* Point)
	{
		point_set_t::const_iterator point = point_set.find(Point);
		if(point != point_set.end())
			return true;

		return false;
	}

	point_set_t& point_set;
};

/// Deletes selected components and their parent components
struct delete_components
{
	delete_components(k3d::mesh& Mesh) :
		mesh(Mesh)
	{
	}

	void operator()(k3d::polyhedron* Polyhedron)
	{
		k3d::polyhedron& polyhedron = *Polyhedron;

		// Find selected faces and save touched points
		point_set_t touched_points;
		for(k3d::polyhedron::faces_t::iterator face = polyhedron.faces.begin(); face != polyhedron.faces.end(); ++face)
			{
				// Find selected faces and faces containing selected edges or vertices
				bool delete_face = false;
				if((*face)->selection_weight)
					delete_face = true;
				else
					{
						// Check for selected edges
						k3d::split_edge* edge = (*face)->first_edge;
						do
							{
								// If edge (or its companion) or its vertex is selected, delete face
								if(edge->selection_weight || (edge->companion && edge->companion->selection_weight) || edge->vertex->selection_weight)
									{
										delete_face = true;
										break;
									}

								edge = edge->face_clockwise;
							}
						while(edge != (*face)->first_edge);
					}

				if(!delete_face)
					continue;

				// Save touched points and select companion edges
				k3d::split_edge* edge = (*face)->first_edge;
				do
					{
						touched_points.insert(edge->vertex);

						// Make sure companion is selected (edge selection doesn't currently select companion)
						if(edge->companion && edge->selection_weight)
							edge->companion->selection_weight = 1.0;

						edge = edge->face_clockwise;
					}
				while(edge != (*face)->first_edge);

				// Save holes touched points
				for(k3d::face::holes_t::iterator hole = (*face)->holes.begin(); hole != (*face)->holes.end(); ++hole)
					{
						for(k3d::split_edge* edge = *hole; edge; edge = edge->face_clockwise)
							{
								touched_points.insert(edge->vertex);

								if(edge->face_clockwise == (*hole))
									break;
							}
					}

				// Delete face
				delete *face;
				*face = 0;
			}

		// Erase deleted faces
		polyhedron.faces.erase(std::remove_if(polyhedron.faces.begin(), polyhedron.faces.end(), std::bind2nd(std::equal_to<k3d::face*>(), 0)), polyhedron.faces.end());

		// Delete floating points
		k3d::for_each_component(mesh, remove_used_points(touched_points));
		if(touched_points.size())
			{
				mesh.points.erase(std::remove_if(mesh.points.begin(), mesh.points.end(), is_in_point_set(touched_points)), mesh.points.end());
				std::for_each(touched_points.begin(), touched_points.end(), k3d::delete_object());
			}

		// Sanity check
		assert_warning(is_valid(polyhedron));
	}

	k3d::mesh& mesh;
};

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// delete_components_implementation

class delete_components_implementation :
	public k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > >
{
	typedef k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > > base;

public:
	delete_components_implementation(k3d::idocument& Document) :
		base(Document)
	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &delete_components_implementation::on_reset_geometry));
		m_mesh_selection.changed_signal().connect(sigc::mem_fun(*this, &delete_components_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &delete_components_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		k3d::replace_selection(m_mesh_selection.value(), *output);

		std::for_each(output->polyhedra.begin(), output->polyhedra.end(), detail::delete_components(*output));

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<delete_components_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink > > > factory(
					k3d::uuid(0xc98c37d5, 0xa50c43c7, 0xb88c3687, 0x73cd3c4d),
				"Delete",
				"Deletes selected faces, edges and vertices",
				"Mesh",
				k3d::iplugin_factory::STABLE);

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// delete_components_factory

k3d::iplugin_factory& delete_components_factory()
{
	return delete_components_implementation::get_factory();
}

} // namespace libk3dmesh


