// K-3D
// Copyright (c) 2005, Romain Behar
//
// Contact: romainbehar@yahoo.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar <romainbehar@yahoo.com>
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/mesh_selection_sink.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/utility.h>

#include "helpers.h"

namespace libk3dmesh
{

namespace detail
{

typedef std::vector<k3d::split_edge*> edges_t;

/// Connects two vertices belonging to the same face
k3d::face* connect_vertices(k3d::face* Face, k3d::split_edge* Edge1, k3d::split_edge* Edge2, edges_t& NewEdges)
{
	// Make new edge loops
	k3d::split_edge* new_edge1 = new k3d::split_edge(Edge1->vertex, Edge2);
	k3d::split_edge* new_edge2 = new k3d::split_edge(Edge2->vertex, Edge1);
	NewEdges.push_back(new_edge1);
	NewEdges.push_back(new_edge2);

	k3d::split_edge* anti_edge1 = k3d::face_anticlockwise(Edge1);
	k3d::split_edge* anti_edge2 = k3d::face_anticlockwise(Edge2);

	anti_edge1->face_clockwise = new_edge1;
	anti_edge2->face_clockwise = new_edge2;

	k3d::join_edges(*new_edge1, *new_edge2);

	// Create a second face for ear cut
	k3d::face* new_face = 0;
	if(helpers::edge_in_face(Face->first_edge, Edge1))
	{
		new_face = new k3d::face(Edge2, Face->material);
	}
	else
	{
		new_face = new k3d::face(Edge1, Face->material);
	}

	return new_face;
}

/// Checks whether one or several colinear edges already link given points
// be careful with parameters order; return true if a line links the two vertices, false otherwise
bool colinear_linked(k3d::split_edge* Edge1, k3d::split_edge* Edge2)
{
	if(Edge1->face_clockwise == Edge2)
		return true;

	k3d::split_edge* edge = Edge1;
	do
	{
		k3d::vector3 vector1 = edge->face_clockwise->vertex->position - edge->vertex->position;
		k3d::vector3 vector2 = Edge2->vertex->position - edge->vertex->position;
		k3d::vector3 normal = vector1 ^ vector2;

		if(normal.Length2() > 1e-6)
		{
			return false;
		}

		edge = edge->face_clockwise;
	}
	while(edge != Edge2);

	return true;
}

/// Connects face's selected vertices
typedef std::vector<k3d::face*> faces_t;
void connect_face_vertices(k3d::face* Face, edges_t& NewEdges, faces_t& NewFaces)
{
	// Get edges with a selected vertex
	edges_t selected_edges;

	k3d::split_edge* edge = Face->first_edge;
	do
	{
		if(edge->vertex->selection_weight)
			selected_edges.push_back(edge);

		edge = edge->face_clockwise;
	}
	while(edge != Face->first_edge);

	// Link non-consecutive points
	while(selected_edges.size() >= 2)
	{
		// Create a triangle linking 3 remaining vertices
		if(selected_edges.size() == 3)
		{
			if(!colinear_linked(selected_edges[0], selected_edges[1]))
				NewFaces.push_back(connect_vertices(Face, selected_edges[0], selected_edges[1], NewEdges));
			if(!colinear_linked(selected_edges[1], selected_edges[2]))
				NewFaces.push_back(connect_vertices(Face, selected_edges[1], selected_edges[2], NewEdges));
			if(!colinear_linked(selected_edges[2], selected_edges[0]))
				NewFaces.push_back(connect_vertices(Face, selected_edges[2], selected_edges[0], NewEdges));

			return;
		}

		// Do an ear-cut on consecutive selected vertex pairs
		unsigned long first_point = 0;
		unsigned long second_point = 1;
		bool found = false;
		do
		{
			k3d::split_edge* edge1 = selected_edges[first_point];
			k3d::split_edge* edge2 = selected_edges[second_point % selected_edges.size()];

			if(edge1->face_clockwise != edge2 && edge2->face_clockwise != edge1)
					// When there are 2 selected edges, edge2 might come before edge1
			{
				// Make sure vectors are not colinear
				k3d::vector3 vector1 = edge1->face_clockwise->vertex->position - edge1->vertex->position;
				k3d::vector3 vector2 = edge2->vertex->position - edge1->vertex->position;
				k3d::vector3 normal = vector1 ^ vector2;

				if(normal.Length2() > 1e-6)
				{
					found = true;
					break;
				}
			}

			++first_point;
			++second_point;
		}
		while(second_point <= selected_edges.size());

		if(!found)
			break;

		// Link points and remove them from the list
		NewFaces.push_back(connect_vertices(Face, selected_edges[first_point], selected_edges[second_point], NewEdges));

		selected_edges.erase(selected_edges.begin() + second_point);
		selected_edges.erase(selected_edges.begin() + first_point);
	}
}

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// connect_vertices_implementation

class connect_vertices_implementation :
	public k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > >
{
	typedef k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > > base;

public:
	connect_vertices_implementation(k3d::idocument& Document) :
		base(Document)
	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &connect_vertices_implementation::on_reset_geometry));
		m_mesh_selection.changed_signal().connect(sigc::mem_fun(*this, &connect_vertices_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &connect_vertices_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		// Clear cache
		m_new_edges.clear();

		// Create output geometry
		std::auto_ptr<k3d::mesh> output(new k3d::mesh());
		k3d::deep_copy(*input, *output);
		k3d::replace_selection(m_mesh_selection.value(), *output);

		// Connect face vertices
		for(k3d::mesh::polyhedra_t::iterator polyhedron = output->polyhedra.begin(); polyhedron != output->polyhedra.end(); ++polyhedron)
		{
			detail::faces_t new_faces;

			// For each face
			for(k3d::polyhedron::faces_t::iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
			{
				detail::connect_face_vertices(*face, m_new_edges, new_faces);
			}

			if(new_faces.size())
				(*polyhedron)->faces.insert((*polyhedron)->faces.end(), new_faces.begin(), new_faces.end());
		}

		// Update output selection
		for(detail::edges_t::iterator edge = m_new_edges.begin(); edge != m_new_edges.end(); ++edge)
		{
			(*edge)->selection_weight = 1.0;
			(*edge)->visible_selection = true;
		}

		return output.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<connect_vertices_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xee6b7115, 0xedba43a2, 0x92750f10, 0x113355a5),
				"ConnectVertices",
				"Creates an edge between selected vertices",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	detail::edges_t m_new_edges;
};

/////////////////////////////////////////////////////////////////////////////
// connect_vertices_factory

k3d::iplugin_factory& connect_vertices_factory()
{
	return connect_vertices_implementation::get_factory();
}

} // namespace libk3dmesh

