// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, read to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the JPEG reader K-3D object, which reads bitmap images using the libjpeg API
		\author Timothy M. Shead
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/ibitmap_read_format.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/ifile_format.h>
#include <k3dsdk/module.h>
#include <k3dsdk/string_modifiers.h>

#include <boost/filesystem/path.hpp>

extern "C"
{

#include <jpeglib.h>

} // extern "C"

namespace libk3djpeg
{

/////////////////////////////////////////////////////////////////////////////
// jpeg_reader

class jpeg_reader :
	public k3d::ifile_format,
	public k3d::ibitmap_read_format,
	public k3d::ideletable
{
public:
	jpeg_reader()
	{
	}

	unsigned long priority()
	{
		return 128;
	}

	bool query_can_handle(const boost::filesystem::path& File)
	{
		return_val_if_fail(!File.empty(), false);
		return "jpeg" == k3d::file_extension(File) || "jpg" == k3d::file_extension(File);
	}

	bool read_file(const boost::filesystem::path& File, k3d::bitmap& Bitmap)
	{
		// Sanity checks ...
		return_val_if_fail(!File.empty(), false);

		k3d::log() << info << "Read " << File.native_file_string() << " using JPEGReader" << std::endl;

		FILE* const file = fopen(File.native_file_string().c_str(), "rb");
		if(!file)
			{
				k3d::log() << error << "Error opening [" << File.native_file_string() << "] for JPEG input" << std::endl;
				return false;
			}

		struct jpeg_decompress_struct cinfo;
		struct jpeg_error_mgr jerr;
		cinfo.err = jpeg_std_error(&jerr);
		jpeg_create_decompress(&cinfo);
		jpeg_stdio_src(&cinfo, file);
		jpeg_read_header(&cinfo, TRUE);

		// Load the input bitmap using 8-bit-integer samples before converting to K-3D's internal format ...
		typedef k3d::basic_rgb<boost::uint8_t> integer_pixel_t;
		typedef k3d::basic_bitmap<integer_pixel_t> integer_bitmap_t;
		integer_bitmap_t bitmap(cinfo.image_width, cinfo.image_height);

		JSAMPROW row = reinterpret_cast<JSAMPLE*>(bitmap.data());
		jpeg_start_decompress(&cinfo);
		while(cinfo.output_scanline < cinfo.output_height)
			{
				jpeg_read_scanlines(&cinfo, &row, 1);
				row += bitmap.width() * 3;
			}
		jpeg_finish_decompress(&cinfo);

		jpeg_destroy_decompress(&cinfo);
		fclose(file);

		Bitmap = bitmap;
		return true;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::application_plugin<jpeg_reader>,
			k3d::interface_list<k3d::ibitmap_read_format> > factory(
				k3d::uuid(0xfb924031, 0x25c242af, 0xa2e1398e, 0x35000e3c),
				"JPEGReader",
				_("JPEG (*.jpeg)"),
				"Bitmap BitmapReader");

		return factory;
	}

private:

};

/////////////////////////////////////////////////////////////////////////////
// jpeg_reader_factory

k3d::iplugin_factory& jpeg_reader_factory()
{
	return jpeg_reader::get_factory();
}

} // namespace libk3djpeg


