// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the OBJReader that imports Wavefront .obj files
		\author Tim Shead (tshead@k-3d.com)
		\author Wladyslaw Strugala (fizws@julia.univ.gda.pl)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include "helpers.h"
#include "obj_helpers.h"

#include <k3dsdk/i18n.h>
#include <k3dsdk/file_helpers.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/idocument.h>
#include <k3dsdk/idocument_read_format.h>
#include <k3dsdk/ifile_format.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/module.h>
#include <k3dsdk/result.h>
#include <k3dsdk/string_modifiers.h>
#include <k3dsdk/utility.h>

#include <k3dsdk/fstream.h>
#include <boost/filesystem/path.hpp>

namespace libk3dgeometry
{

/////////////////////////////////////////////////////////////////////////////
// obj_reader_implementation

class obj_reader_implementation :
	public k3d::ifile_format,
	public k3d::idocument_read_format,
	public k3d::ideletable
{
public:
	obj_reader_implementation()
	{
	}

	unsigned long priority()
	{
		return 128;
	}

	bool query_can_handle(const boost::filesystem::path& FilePath)
	{
		return "obj" == k3d::file_extension(FilePath);
	}

	bool read_file(k3d::idocument& Document, const boost::filesystem::path& FilePath)
	{
		// Create document object ...
		k3d::inode* frozen_mesh;
		k3d::inode* instance;
		k3d::mesh* const mesh = detail::create_mesh(Document, "OBJ import", frozen_mesh, instance);
		return_val_if_fail(mesh, false);

		try
		{
			if(!detail::load_obj_file(*mesh, Document, FilePath, "Texture U", "Texture V", "Texture W"))
				return false;
		}
		catch(std::exception& e)
		{
			k3d::log() << error << "error loading " << FilePath.native_file_string() << std::endl;
			return false;
		}

		return true;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::application_plugin<obj_reader_implementation>, k3d::interface_list<k3d::idocument_read_format> > factory(
			k3d::uuid(0x45a20d5f, 0xd2f447a9, 0x9d772381, 0xac833c39),
			"OBJReader",
			_("Wavefront ( .obj )"),
			"GeometryReader");

		return factory;
	}
};

k3d::iplugin_factory& obj_reader_factory()
{
	return obj_reader_implementation::get_factory();
}

} // namespace libk3dgeometry


