// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/axis.h>
#include <k3dsdk/bounding_box.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/mesh_selection_sink.h>
#include <k3dsdk/module.h>

#include <iterator>

namespace libk3ddeformation
{

/////////////////////////////////////////////////////////////////////////////
// taper_points

class taper_points :
	public k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > >
{
	typedef k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > > base;

public:
	taper_points(k3d::idocument& Document) :
		base(Document),
		m_axis(init_owner(*this) + init_name("axis") + init_label(_("Axis")) + init_description(_("Taper along this axis")) + init_value(k3d::Y) + init_enumeration(k3d::axis_values())),
		m_taper_factor(init_owner(*this) + init_name("taper_fator") + init_label(_("Taper fator")) + init_description(_("Taper factor")) + init_value(0.0) + init_precision(2) + init_step_increment(k3d::radians(1.0)) + init_units(typeid(k3d::measurement::scalar))),
		m_displace_x(init_owner(*this) + init_name("displace_x") + init_label(_("Displace X")) + init_description(_("Displacement on X axis")) + init_value(true)),
		m_displace_y(init_owner(*this) + init_name("displace_y") + init_label(_("Displace Y")) + init_description(_("Displacement on Y axis")) + init_value(false)),
		m_displace_z(init_owner(*this) + init_name("displace_z") + init_label(_("Displace Z")) + init_description(_("Displacement on Z axis")) + init_value(true))
	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &taper_points::on_reset_geometry));

		m_axis.changed_signal().connect(sigc::mem_fun(*this, &taper_points::on_reshape_geometry));
		m_mesh_selection.changed_signal().connect(sigc::mem_fun(*this, &taper_points::on_reset_geometry));

		m_taper_factor.changed_signal().connect(sigc::mem_fun(*this, &taper_points::on_reshape_geometry));
		m_displace_x.changed_signal().connect(sigc::mem_fun(*this, &taper_points::on_reshape_geometry));
		m_displace_y.changed_signal().connect(sigc::mem_fun(*this, &taper_points::on_reshape_geometry));
		m_displace_z.changed_signal().connect(sigc::mem_fun(*this, &taper_points::on_reshape_geometry));

		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &taper_points::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		const k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return;

		k3d::mesh* const output = const_cast<k3d::mesh*>(m_output_mesh.internal_value());
		if(!output)
			return;

		reshape_geometry(*input, *output);

		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input mesh and modify the copy ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		k3d::replace_selection(m_mesh_selection.value(), *output);
		reshape_geometry(*input, *output);

		return output;
	}

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		// Sanity checks ...
		assert(Source.points.size() == Target.points.size());

		const k3d::bounding_box3 bounds = k3d::bounds(Source);

		const k3d::axis axis = m_axis.value();
		const double taper_factor = m_taper_factor.value();
		const bool displace_x = m_displace_x.value();
		const bool displace_y = m_displace_y.value();
		const bool displace_z = m_displace_z.value();

		double size = 0.0;
		switch(axis)
			{
				case k3d::X:
					size = std::max(std::abs(bounds.px), std::abs(bounds.nx));
					break;
				case k3d::Y:
					size = std::max(std::abs(bounds.py), std::abs(bounds.ny));
					break;
				case k3d::Z:
					size = std::max(std::abs(bounds.pz), std::abs(bounds.nz));
					break;
				default:
					assert_not_reached();
			}
		if(0.0 == size)
			return;

		for(unsigned long i = 0; i != Target.points.size(); ++i)
			{
				const double scale = k3d::mix(1.0, 1.0 - taper_factor, std::abs(Source.points[i]->position[axis] / size));

				k3d::vector3 position(Source.points[i]->position);
				if(displace_x)
					position[0] *= scale;
				if(displace_y)
					position[1] *= scale;
				if(displace_z)
					position[2] *= scale;

				Target.points[i]->position = k3d::mix(Source.points[i]->position, position, Target.points[i]->selection_weight);
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<taper_points>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x4d05f002, 0x27764b67, 0xa1a12e21, 0x436c3d06),
				"TaperPoints",
				_("Tapers mesh points along an axis"),
				"Deformation",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(k3d::axis, immutable_name, change_signal, with_undo, local_storage, no_constraint, enumeration_property, with_serialization) m_axis;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_taper_factor;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_displace_x;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_displace_y;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_displace_z;
};

/////////////////////////////////////////////////////////////////////////////
// taper_points_factory

k3d::iplugin_factory& taper_points_factory()
{
	return taper_points::get_factory();
}

} // namespace libk3ddeformation


