// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements a deformation bone primitive for use with skeletons & kinematics
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Brett W. McCoy (bmccoy@chapelperilous.net)
*/

#include <k3dsdk/bounded.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transformable.h>

namespace libk3ddeformation
{

namespace detail
{

/// Functor object for applying a transformation to a collection of points
struct transform_points
{
	transform_points(const k3d::matrix4& Matrix) :
		matrix(Matrix)
	{
	}

	void operator()(k3d::point* const Point)
	{
		Point->position = matrix * Point->position;
	}

	const k3d::matrix4 matrix;
};

}

/////////////////////////////////////////////////////////////////////////////
// deformation_bone

class deformation_bone :
	public k3d::bounded<k3d::gl::drawable<k3d::mesh_modifier<k3d::transformable<k3d::persistent<k3d::node> > > > >
{
	typedef k3d::bounded<k3d::gl::drawable<k3d::mesh_modifier<k3d::transformable<k3d::persistent<k3d::node> > > > > base;

public:
	deformation_bone(k3d::idocument& Document) :
		base(Document),
		m_length(init_owner(*this) + init_name("length") + init_label(_("Length")) + init_description(_("Length")) + init_value(5.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance)))

	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &deformation_bone::on_reset_geometry));
		m_input_matrix.changed_signal().connect(sigc::mem_fun(*this, &deformation_bone::on_reset_geometry));

		m_length.changed_signal().connect(sigc::mem_fun(*this, &deformation_bone::async_redraw_all));

		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &deformation_bone::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		// Transform the points ...
		std::for_each(output->points.begin(), output->points.end(), detail::transform_points(matrix()));

		return output;
	}

	const k3d::bounding_box3 extents()
	{
		const double length = m_length.value();
		return k3d::bounding_box3(.5, -.5, length, -.5, .5, -.5);
	}

	void draw(const double length)
	{
		glBegin(GL_TRIANGLE_FAN);
		glVertex3d(0, 0, length);
		glVertex3d(0, 0.5, 0);
		glVertex3d(0.5, 0, 0);
		glVertex3d(0, -0.5, 0);
		glVertex3d(-0.5, 0, 0);
		glVertex3d(0, 0.5, 0);
		glEnd();

		glBegin(GL_TRIANGLE_FAN);
		glVertex3d(0, 0, length * -0.1);
		glVertex3d(0, 0.5, 0);
		glVertex3d(0.5, 0, 0);
		glVertex3d(0, -0.5, 0);
		glVertex3d(-0.5, 0, 0);
		glVertex3d(0, 0.5, 0);
		glEnd();
	}

	void on_gl_draw(const k3d::gl::render_state& State)
	{
		glColor3dv(get_visible_selection() ? k3d::vector3(1, 1, 1) : k3d::vector3(0, 0, 0));
		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
		glDisable(GL_TEXTURE_1D);
		glDisable(GL_TEXTURE_2D);
		glDisable(GL_LIGHTING);
		glDisable(GL_AUTO_NORMAL);
		glDisable(GL_CULL_FACE);

		draw(m_length.value());
	}

	void on_gl_select(const k3d::gl::render_state& State, const k3d::gl::select_state& SelectState)
	{
		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		glDisable(GL_TEXTURE_1D);
		glDisable(GL_TEXTURE_2D);
		glDisable(GL_LIGHTING);
		glDisable(GL_AUTO_NORMAL);
		glDisable(GL_CULL_FACE);

		k3d::gl::push_selection_token(this);
		draw(m_length.value());
		k3d::gl::pop_selection_token();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<deformation_bone>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink> > > factory(
				k3d::uuid(0x54e46452, 0x681f4ed6, 0xb30832d5, 0x06a69295),
				"DeformationBone",
				"Deformation Bone",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_length;
};

/////////////////////////////////////////////////////////////////////////////
// deformation_bone_factory

k3d::iplugin_factory& deformation_bone_factory()
{
	return deformation_bone::get_factory();
}

} // namespace libk3ddeformation


