// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/axis.h>
#include <k3dsdk/bounding_box.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_modifier.h>
#include <k3dsdk/mesh_selection_sink.h>
#include <k3dsdk/module.h>

#include <iterator>

namespace libk3ddeformation
{

/////////////////////////////////////////////////////////////////////////////
// center_points

class center_points :
	public k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > >
{
	typedef k3d::mesh_selection_sink<k3d::mesh_modifier<k3d::persistent<k3d::node> > > base;

public:
	center_points(k3d::idocument& Document) :
		base(Document),
		m_center_x(init_owner(*this) + init_name("center_x") + init_label(_("Center X")) + init_description(_("Center X components around origin")) + init_value(true)),
		m_center_y(init_owner(*this) + init_name("center_y") + init_label(_("Center Y")) + init_description(_("Center Y components around origin")) + init_value(true)),
		m_center_z(init_owner(*this) + init_name("center_z") + init_label(_("Center Z")) + init_description(_("Center Z components around origin")) + init_value(true))
	{
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &center_points::on_reset_geometry));
		m_mesh_selection.changed_signal().connect(sigc::mem_fun(*this, &center_points::on_reset_geometry));

		m_center_x.changed_signal().connect(sigc::mem_fun(*this, &center_points::on_reshape_geometry));
		m_center_y.changed_signal().connect(sigc::mem_fun(*this, &center_points::on_reshape_geometry));
		m_center_z.changed_signal().connect(sigc::mem_fun(*this, &center_points::on_reshape_geometry));

		m_output_mesh.need_data_signal().connect(sigc::mem_fun(*this, &center_points::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	void on_reshape_geometry()
	{
		if(m_output_mesh.empty())
			return;

		const k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return;

		k3d::mesh* const output = const_cast<k3d::mesh*>(m_output_mesh.internal_value());
		if(!output)
			return;

		reshape_geometry(*input, *output);

		m_output_mesh.changed_signal().emit();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input mesh and modify the copy ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);
		k3d::replace_selection(m_mesh_selection.value(), *output);
		reshape_geometry(*input, *output);

		return output;
	}

	void reshape_geometry(const k3d::mesh& Source, k3d::mesh& Target)
	{
		// Sanity checks ...
		assert(Source.points.size() == Target.points.size());

		const k3d::bounding_box3 bounds = k3d::bounds(Source);
		if(bounds.empty())
			return;

		const bool center_x = m_center_x.value();
		const bool center_y = m_center_y.value();
		const bool center_z = m_center_z.value();

		const k3d::vector3 offset(
			center_x ? -0.5 * (bounds.px + bounds.nx) : 0,
			center_y ? -0.5 * (bounds.py + bounds.ny) : 0,
			center_z ? -0.5 * (bounds.pz + bounds.nz) : 0);

		for(unsigned long i = 0; i != Target.points.size(); ++i)
			Target.points[i]->position = k3d::mix(Source.points[i]->position, Source.points[i]->position + offset, Target.points[i]->selection_weight);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<center_points>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xc030994f, 0x97ed4165, 0x9ac0b7f0, 0xa777abbc),
				"CenterPoints",
				_("Centers mesh points around the origin"),
				"Mesh",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_center_x;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_center_y;
	k3d_data(bool, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_center_z;
};

/////////////////////////////////////////////////////////////////////////////
// center_points_factory

k3d::iplugin_factory& center_points_factory()
{
	return center_points::get_factory();
}

} // namespace libk3ddeformation


