// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the Sphere K-3D object, which models a RenderMan sphere primitive
		\author Tim Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/algebra.h>
#include <k3dsdk/bounded.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/geometry.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/iapplication.h>
#include <k3dsdk/idag.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/material_client.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/renderman.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transformable.h>

#include <boost/scoped_ptr.hpp>

namespace
{

/////////////////////////////////////////////////////////////////////////////
// sphere

class sphere :
	public k3d::bounded<k3d::material_client<k3d::gl::drawable<k3d::ri::renderable<k3d::transformable<k3d::persistent<k3d::node> > > > > >
{
	typedef k3d::bounded<k3d::material_client<k3d::gl::drawable<k3d::ri::renderable<k3d::transformable<k3d::persistent<k3d::node> > > > > > base;

public:
	sphere(k3d::idocument& Document) :
		base(Document),
		m_radius(init_owner(*this) + init_name("radius") + init_label(_("Radius")) + init_description(_("Sphere radius")) + init_value(5.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_zmin(init_owner(*this) + init_name("zmin") + init_label(_("Z min")) + init_description(_("From RenderMan specification")) + init_value(-1.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_zmax(init_owner(*this) + init_name("zmax") + init_label(_("Z max")) + init_description(_("From RenderMan specification")) + init_value(1.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_thetamax(init_owner(*this) + init_name("thetamax") + init_label(_("Theta max")) + init_description(_("From RenderMan specification")) + init_value(k3d::radians(360.0)) + init_precision(2) + init_step_increment(k3d::radians(1.0)) + init_units(typeid(k3d::measurement::angle)))
	{
		m_radius.changed_signal().connect(sigc::mem_fun(*this, &sphere::reset_geometry));
		m_zmin.changed_signal().connect(sigc::mem_fun(*this, &sphere::reset_geometry));
		m_zmax.changed_signal().connect(sigc::mem_fun(*this, &sphere::reset_geometry));
		m_thetamax.changed_signal().connect(sigc::mem_fun(*this, &sphere::reset_geometry));

		m_visible_selection.changed_signal().connect(sigc::mem_fun(*this, &sphere::async_redraw_all));
		m_input_matrix.changed_signal().connect(sigc::mem_fun(*this, &sphere::async_redraw_all));
	}

	const k3d::bounding_box3 extents()
	{
		const double radius = m_radius.value();
		const double zmax = m_zmax.value();
		const double zmin = m_zmin.value();

		return k3d::bounding_box3(radius, -radius, radius, -radius, zmax * radius, zmin * radius);
	}

	void reset_geometry()
	{
		m_gl_control_points.clear();
		k3d::gl::redraw_all(document(), k3d::gl::irender_engine::ASYNCHRONOUS);
	}

	void draw(const nurbs_renderer_t Nurbs)
	{
		if(m_gl_control_points.empty())
		{
			const double radius = m_radius.value();
			const double zmin = m_zmin.value();
			const double zmax = m_zmax.value();
			const double thetamax = m_thetamax.value();

			const double phimin = (zmin > -1) ? asin(zmin) : -k3d::pi_over_2();
			const double phimax = (zmax < 1) ? asin(zmax) : k3d::pi_over_2();

			std::vector<double> v_weights;
			std::vector<k3d::vector3> v_arc_points;
			k3d::nurbs_arc(k3d::vector3(0, 1, 0), k3d::vector3(0, 0, 1), phimin, phimax, 2, m_gl_v_knot_vector, v_weights, v_arc_points);

			std::vector<double> u_weights;
			std::vector<k3d::vector3> u_arc_points;
			k3d::nurbs_arc(k3d::vector3(1, 0, 0), k3d::vector3(0, 1, 0), 0, thetamax, 4, m_gl_u_knot_vector, u_weights, u_arc_points);

			for(unsigned long v = 0; v != v_arc_points.size(); ++v)
			{
				const k3d::vector3 offset = radius * v_arc_points[v][2] * k3d::vector3(0, 0, 1);
				const double radius2 = radius * v_arc_points[v][1];
				const double v_weight = v_weights[v];

				for(unsigned long u = 0; u != u_arc_points.size(); ++u)
				{
					m_gl_control_points.push_back(v_weight * u_weights[u] * (radius2 * u_arc_points[u][0] + offset[0]));
					m_gl_control_points.push_back(v_weight * u_weights[u] * (radius2 * u_arc_points[u][1] + offset[1]));
					m_gl_control_points.push_back(v_weight * u_weights[u] * (radius2 * u_arc_points[u][2] + offset[2]));
					m_gl_control_points.push_back(v_weight * u_weights[u]);
				}
			}
		}

		gluBeginSurface(Nurbs);
		gluNurbsSurface(Nurbs, m_gl_u_knot_vector.size(), &m_gl_u_knot_vector[0], m_gl_v_knot_vector.size(), &m_gl_v_knot_vector[0], 4, 36, &m_gl_control_points[0], 3, 3, GL_MAP2_VERTEX_4);
		gluEndSurface(Nurbs);
	}

	void on_gl_draw(const k3d::gl::render_state& State)
	{
		k3d::gl::setup_material(m_material.value());

		const nurbs_renderer_t nurbs = nurbs_renderer(State);

		glColor3dv(get_visible_selection() ? k3d::vector3(1, 1, 1) : k3d::vector3(0, 0, 0));
		gluNurbsProperty(nurbs, GLU_DISPLAY_MODE, GLU_OUTLINE_PATCH);
		glDisable(GL_LIGHTING);
		glDisable(GL_AUTO_NORMAL);
		draw(nurbs);

		if(!State.draw_two_sided)
			glEnable(GL_CULL_FACE);

		gluNurbsProperty(nurbs, GLU_DISPLAY_MODE, GLU_FILL);
		glEnable(GL_LIGHTING);
		glEnable(GL_AUTO_NORMAL);
		glPolygonOffset(1.0, 1.0);
		glEnable(GL_POLYGON_OFFSET_FILL);
		draw(nurbs);
		glDisable(GL_POLYGON_OFFSET_FILL);
	}

	void on_gl_select(const k3d::gl::render_state& State, const k3d::gl::select_state& SelectState)
	{
		const nurbs_renderer_t nurbs = nurbs_renderer(State);
		gluNurbsProperty(nurbs, GLU_DISPLAY_MODE, GLU_FILL);
		glDisable(GL_LIGHTING);
		glDisable(GL_AUTO_NORMAL);
		glDisable(GL_CULL_FACE);

		k3d::gl::push_selection_token(this);
		draw(nurbs);
		k3d::gl::pop_selection_token();
	}

	void on_renderman_render(const k3d::ri::render_state& State)
	{
		const double radius = m_radius.value();
		const double zmin = m_zmin.value();
		const double zmax = m_zmax.value();
		const double thetamax = k3d::degrees(m_thetamax.value());

		k3d::ri::setup_material(m_material.value(), State);
		State.engine.RiSphereV(radius, radius * zmin, radius * zmax, thetamax);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<sphere>,
			k3d::interface_list<k3d::itransform_source,
			k3d::interface_list<k3d::itransform_sink > > > factory(
				k3d::classes::Sphere(),
				"Sphere",
				_("Sphere primitive"),
				"Conics",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_radius;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_zmin;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_zmax;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_thetamax;

	std::vector<GLfloat> m_gl_u_knot_vector;
	std::vector<GLfloat> m_gl_v_knot_vector;
	std::vector<GLfloat> m_gl_control_points;
};

} // namespace

namespace libk3dcore
{

/////////////////////////////////////////////////////////////////////////////
// sphere_factory

k3d::iplugin_factory& sphere_factory()
{
	return sphere::get_factory();
}

} // namespace libk3dcore


