// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the Paraboloid K-3D object, which renders a RenderMan paraboloid primitive
		\author Tim Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/algebra.h>
#include <k3dsdk/bounded.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/gl.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/iapplication.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/material_client.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/renderman.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transformable.h>

namespace
{

/////////////////////////////////////////////////////////////////////////////
// paraboloid

class paraboloid :
	public k3d::material_client<k3d::bounded<k3d::gl::drawable<k3d::ri::renderable<k3d::transformable<k3d::persistent<k3d::node> > > > > >
{
	typedef k3d::material_client<k3d::bounded<k3d::gl::drawable<k3d::ri::renderable<k3d::transformable<k3d::persistent<k3d::node> > > > > > base;

public:
	paraboloid(k3d::idocument& Document) :
		base(Document),
		m_radius(init_owner(*this) + init_name("radius") + init_label(_("Radius")) + init_description(_("Radius")) + init_value(5.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_zmax(init_owner(*this) + init_name("zmax") + init_label(_("Z max")) + init_description(_("Maximum point position")) + init_value(10.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_zmin(init_owner(*this) + init_name("zmin") + init_label(_("Z min")) + init_description(_("Origin position")) + init_value(0.0) + init_precision(2) + init_step_increment(0.1) + init_units(typeid(k3d::measurement::distance))),
		m_thetamax(init_owner(*this) + init_name("thetamax") + init_label(_("Theta max")) + init_description(_("From RenderMan specification")) + init_value(k3d::radians(360.0)) + init_precision(2) + init_step_increment(k3d::radians(1.0)) + init_units(typeid(k3d::measurement::angle)))
	{
		m_radius.changed_signal().connect(sigc::mem_fun(*this, &paraboloid::async_redraw_all));
		m_zmin.changed_signal().connect(sigc::mem_fun(*this, &paraboloid::async_redraw_all));
		m_zmax.changed_signal().connect(sigc::mem_fun(*this, &paraboloid::async_redraw_all));
		m_thetamax.changed_signal().connect(sigc::mem_fun(*this, &paraboloid::async_redraw_all));

		m_visible_selection.changed_signal().connect(sigc::mem_fun(*this, &paraboloid::async_redraw_all));
		m_input_matrix.changed_signal().connect(sigc::mem_fun(*this, &paraboloid::async_redraw_all));
	}

	const k3d::bounding_box3 extents()
	{
		const double radius = m_radius.value();
		const double zmin = m_zmin.value();
		const double zmax = m_zmax.value();

		return k3d::bounding_box3(radius, -radius, radius, -radius, zmax, zmin);
	}

	void draw()
	{
		const double radius = m_radius.value();
		const double zmin = std::max(0.0, m_zmin.value());
		const double zmax = m_zmax.value();
		const double thetamax = m_thetamax.value();

		const unsigned long nsides = 16;
		const unsigned long nlevels = 8;

		const double start = 0;
		const double step = thetamax / static_cast<double>(nsides);
		for(unsigned long j = 0; j < nlevels; j++)
		{
			const double zmax1 = zmin + (zmax - zmin) * static_cast<double>(j+1)/static_cast<double>(nlevels);
			const double zmax2 = zmin + (zmax - zmin) * static_cast<double>(j)/static_cast<double>(nlevels);
			const double radius1 = radius * sqrt(zmax1 / zmax);
			const double radius2 = std::max(radius * 0.001, radius * sqrt(zmax2 / zmax));

			for(unsigned long i = 0; i < nsides; i++)
			{
				glBegin(GL_POLYGON);
					double current = start + static_cast<double>(i) * step;

					k3d::vector3 p1(radius1*cos(current), radius1*sin(current), zmax1);
					k3d::vector3 p2(radius2*cos(current), radius2*sin(current), zmax2);
					k3d::vector3 p3(radius2*cos(current + step), radius2*sin(current + step), zmax2);
					k3d::vector3 p4(radius1*cos(current + step), radius1*sin(current + step), zmax1);

					glNormal3dv((p3 - p4) ^ (p3 - p2));

					glVertex3dv(p1);
					glVertex3dv(p2);
					glVertex3dv(p3);
					glVertex3dv(p4);

				glEnd();
			}
		}
	}

	void on_gl_draw(const k3d::gl::render_state& State)
	{
		k3d::gl::setup_material(m_material.value());

		glDisable(GL_CULL_FACE);
		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
		glColor3dv(get_visible_selection() ? k3d::vector3(1, 1, 1) : k3d::vector3(0, 0, 0));
		glDisable(GL_LIGHTING);
		draw();

		if(!State.draw_two_sided)
			glEnable(GL_CULL_FACE);

		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		glEnable(GL_LIGHTING);
		glPolygonOffset(1.0, 1.0);
		glEnable(GL_POLYGON_OFFSET_FILL);
		draw();
		glDisable(GL_POLYGON_OFFSET_FILL);
	}

	void on_gl_select(const k3d::gl::render_state& State, const k3d::gl::select_state& SelectState)
	{
		glDisable(GL_LIGHTING);
		glDisable(GL_CULL_FACE);

		k3d::gl::push_selection_token(this);
		draw();
		k3d::gl::pop_selection_token();
	}

	void on_renderman_render(const k3d::ri::render_state& State)
	{
		const double radius = m_radius.value();
		const double zmin = m_zmin.value();
		const double zmax = m_zmax.value();
		const double thetamax = k3d::degrees(m_thetamax.value());

		k3d::ri::setup_material(m_material.value(), State);
		State.engine.RiParaboloidV(radius, zmin, zmax, thetamax);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<paraboloid>,
			k3d::interface_list<k3d::itransform_source,
			k3d::interface_list<k3d::itransform_sink > > > factory(
				k3d::classes::Paraboloid(),
				"Paraboloid",
				_("Paraboloid primitive"),
				"Conics",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_radius;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_zmax;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_zmin;
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, measurement_property, with_serialization) m_thetamax;
};

} // namespace

namespace libk3dcore
{

/////////////////////////////////////////////////////////////////////////////
// paraboloid_factory

k3d::iplugin_factory& paraboloid_factory()
{
	return paraboloid::get_factory();
}

} // namespace libk3dcore


