// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the DocumentReader K-3D plugin, which imports the K-3D native file format
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/algebra.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/data.h>
#include <k3dsdk/fstream.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/idocument.h>
#include <k3dsdk/idocument_read_format.h>
#include <k3dsdk/ifile_format.h>
#include <k3dsdk/inode_collection.h>
#include <k3dsdk/ipersistent.h>
#include <k3dsdk/itransform_sink.h>
#include <k3dsdk/itransform_source.h>
#include <k3dsdk/log.h>
#include <k3dsdk/module.h>
#include <k3dsdk/persistent_lookup.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/serialization.h>
#include <k3dsdk/string_modifiers.h>
#include <k3dsdk/vectors.h>
#include <k3dsdk/xml.h>

#include <iostream>

using namespace k3d::xml;

namespace libk3dcore
{

/////////////////////////////////////////////////////////////////////////////
// document_reader

class document_reader :
	public k3d::ifile_format,
	public k3d::idocument_read_format,
	public k3d::ideletable
{
public:
	unsigned long priority()
	{
		return 128;
	}

	bool query_can_handle(const boost::filesystem::path& FilePath)
	{
		return "k3d" == k3d::file_extension(FilePath);
	}

	bool read_file(k3d::idocument& Document, const boost::filesystem::path& FilePath)
	{
		element xml("k3dml");
		try
		{
			k3d::filesystem::ifstream stream(FilePath);
			hide_progress progress;
			parse(xml, stream, FilePath.native_file_string(), progress);
		}
		catch(std::exception& e)
		{
			k3d::log() << error << e.what() << std::endl;
			return false;
		}

		// Make sure it's a K3D document ...
		return_val_if_fail(xml.name == "k3dml", false);

		// Look for a document version ...
		std::stringstream version(attribute_text(xml, "version"));
		unsigned long major_version = 0;
		unsigned long minor_version = 0;
		unsigned long revision = 0;
		unsigned long build = 0;
		char point;
		version >> major_version >> point >> minor_version >> point >> revision >> point >> build;

		const boost::filesystem::path root_path = FilePath.branch_path();
		k3d::persistent_lookup persistent_lookup;
		k3d::ipersistent::load_context context(root_path, persistent_lookup);

		// Load per-node data ...
		if(element* xml_document = find_element(xml, "document"))
		{
			// Handle documents from older versions of the software by modifying the document
			upgrade_document(*xml_document);

			// Load nodes
			if(element* xml_nodes = find_element(*xml_document, "nodes"))
			{
				k3d::inode_collection::nodes_t nodes;
				std::vector<k3d::ipersistent*> persistent_nodes;
				std::vector<element*> node_storage;

				for(element::elements_t::iterator xml_node = xml_nodes->children.begin(); xml_node != xml_nodes->children.end(); ++xml_node)
				{
					if(xml_node->name != "node")
						continue;

					if(attribute_value<bool>(*xml_node, "do_not_load", false))
						continue;

					const std::string name = attribute_text(*xml_node, "name");
					const k3d::uuid class_id = attribute_value<k3d::uuid>(*xml_node, "class", k3d::uuid::null());
					if(class_id == k3d::uuid::null())
					{
						k3d::log() << error << "node [" << name << "] with unspecified class ID will not be loaded" << std::endl;
						continue;
					}

					const k3d::ipersistent_lookup::id_type node_id = attribute_value<k3d::ipersistent_lookup::id_type>(*xml_node, "id", 0);
					if(node_id == 0)
					{
						k3d::log() << error << "node [" << name << "] with unspecified ID will not be loaded" << std::endl;
						continue;
					}

					k3d::iplugin_factory* const plugin_factory = k3d::plugin(class_id);
					if(!plugin_factory)
					{
						k3d::log() << error << "node [" << name << "] with unknown class ID [" << class_id << "] will not be loaded" << std::endl;
						continue;
					}

					k3d::idocument_plugin_factory* const document_plugin_factory = dynamic_cast<k3d::idocument_plugin_factory*>(plugin_factory);
					if(!document_plugin_factory)
					{
						k3d::log() << error << "Non-document plugin [" << name << "] will not be loaded" << std::endl;
						continue;
					}

					k3d::inode* const node = document_plugin_factory->create_plugin(Document);
					if(!node)
					{
						k3d::log() << error << "Error creating node [" << name << "] instance" << std::endl;
						continue;
					}

					k3d::ipersistent* const persistent = dynamic_cast<k3d::ipersistent*>(node);
					if(!persistent)
					{
						k3d::log() << error << "node [" << name << "] does not support persistence" << std::endl;

						delete dynamic_cast<k3d::ideletable*>(node);
						continue;
					}

					k3d::undoable_new(dynamic_cast<k3d::ideletable*>(node), Document);

					nodes.push_back(node);
					persistent_nodes.push_back(persistent);
					node_storage.push_back(&(*xml_node));

					persistent_lookup.insert_lookup(node_id, node);
				}

				Document.nodes().add_nodes(nodes);

				for(unsigned long i = 0; i != persistent_nodes.size(); ++i)
					persistent_nodes[i]->load(*node_storage[i], context);
			}

			// Load the DAG ...
			k3d::load_dag(Document, *xml_document, context);
		}

		// Load per-plugin-type data ....
		if(element* const xml_application = find_element(xml, "application"))
		{
			if(element* const xml_plugins = find_element(*xml_application, "plugins"))
			{
				for(element::elements_t::iterator xml_plugin = xml_plugins->children.begin(); xml_plugin != xml_plugins->children.end(); ++xml_plugin)
				{
					if(xml_plugin->name != "plugin")
						continue;

					if(attribute_value<bool>(*xml_plugin, "do_not_load", false))
						continue;

					const k3d::uuid class_id = attribute_value<k3d::uuid>(*xml_plugin, "class", k3d::uuid::null());
					if(class_id == k3d::uuid::null())
					{
						k3d::log() << error << "Plugin with unspecified class ID will not be loaded" << std::endl;
						continue;
					}

					if(!Document.plugin_serialization_handlers().count(class_id))
					{
						k3d::log() << error << "Unknown plugin type [" << class_id << "] will not be loaded" << std::endl;
						continue;
					}

					k3d::ipersistent* const handler = Document.plugin_serialization_handlers().find(class_id)->second;
					if(!handler)
					{
						k3d::log() << error << "Invalid serialization handler for plugin type [" << class_id << "] will not be used" << std::endl;
						continue;
					}

					handler->load(*xml_plugin, context);
				}
			}
		}

		return true;
	}

	/// Upgrades XML schema to match newer node and directed graph terminology
	void upgrade_document_schema(element& XMLDocument)
	{
		// Change <objects> to <nodes> ...
		if(element* const xml_objects = find_element(XMLDocument, "objects"))
		{
			k3d::log() << warning << "Converting obsolete <objects> tag to <nodes> tag" << std::endl;
			xml_objects->name = "nodes";
		}

		// Change <object> to <node> ...
		if(element* const xml_nodes = find_element(XMLDocument, "nodes"))
		{
			bool nag_object = true;
		
			for(element::elements_t::iterator xml_node = xml_nodes->children.begin(); xml_node != xml_nodes->children.end(); ++xml_node)
			{
				if(xml_node->name == "object")
				{
					if(nag_object)
					{
						k3d::log() << warning << "Converting obsolete <object> tags to <node> tags" << std::endl;
						nag_object = false;
					}
					
					xml_node->name = "node";
				}
			}
		}

		// Change <variables> to <properties>
		if(element* const xml_nodes = find_element(XMLDocument, "nodes"))
		{
			bool nag_variables = true;

			for(element::elements_t::iterator xml_node = xml_nodes->children.begin(); xml_node != xml_nodes->children.end(); ++xml_node)
			{
				if(xml_node->name != "node")
					continue;

				element* const xml_variables = find_element(*xml_node, "variables");
				if(!xml_variables)
					continue;

				if(nag_variables)
				{
					nag_variables = false;
					k3d::log() << warning << "Converting obsolete <variables> tags to <properties> tags" << std::endl;
				}

				xml_variables->name = "properties";
			}
		}
		
		// Change <variable>, <object>, <shader>, and <argument> to <property>
		if(element* const xml_nodes = find_element(XMLDocument, "nodes"))
		{
			bool nag_object = true;
			bool nag_shader = true;
			bool nag_variable = true;

			for(element::elements_t::iterator xml_node = xml_nodes->children.begin(); xml_node != xml_nodes->children.end(); ++xml_node)
			{
				if(xml_node->name != "node")
					continue;

				element* const xml_properties = find_element(*xml_node, "properties");
				if(!xml_properties)
					continue;

				for(element::elements_t::iterator xml_property = xml_properties->children.begin(); xml_property != xml_properties->children.end(); ++xml_property)
				{
					if(xml_property->name == "object")
					{
						if(nag_object)
						{
							nag_object = false;
							k3d::log() << warning << "Converting obsolete <object> tags to <property> tags" << std::endl;
						}
						xml_property->name = "property";
					}
					else if(xml_property->name == "shader")
					{
						if(nag_shader)
						{
							nag_shader = false;
							k3d::log() << warning << "Converting obsolete <shader> tags to <property> tags" << std::endl;
						}
						xml_property->name = "property";
					}
					else if(xml_property->name == "variable")
					{
						if(nag_variable)
						{
							nag_variable = false;
							k3d::log() << warning << "Converting obsolete <variable> tags to <property> tags" << std::endl;
						}
						xml_property->name = "property";
					}
				}
			}
		}

		// Change <dag> to <dependencies> ...
		if(element* const xml_dag = find_element(XMLDocument, "dag"))
		{
			k3d::log() << warning << "Converting obsolete <dag> tag to <dependencies> tag" << std::endl;
			xml_dag->name = "dependencies";
		}

		// Update <dependency> properties ...
		if(element* const xml_dependencies = find_element(XMLDocument, "dependencies"))
		{
			bool nag_from = true;
			bool nag_to = true;

			for(element::elements_t::iterator xml_dependency = xml_dependencies->children.begin(); xml_dependency != xml_dependencies->children.end(); ++xml_dependency)
			{
				if(xml_dependency->name != "dependency")
					continue;

				if(attribute* const from_object = find_attribute(*xml_dependency, "from_object"))
				{
					if(nag_from)
					{
						nag_from = false;
						k3d::log() << warning << "converting from_object attributes" << std::endl;
					}
					
					from_object->name = "from_node";
				}

				if(attribute* const to_object = find_attribute(*xml_dependency, "to_object"))
				{
					if(nag_to)
					{
						nag_to = false;
						k3d::log() << warning << "converting to_object attributes" << std::endl;
					}
					
					to_object->name = "to_node";
				}
			}
		}
	}
	
	/// Upgrades PolySphere nodes so their "type" property is set correctly
	void upgrade_poly_sphere_nodes(element& XMLDocument)
	{
		element* const xml_nodes = find_element(XMLDocument, "nodes");
		if(!xml_nodes)
			return;

		for(element::elements_t::iterator xml_node = xml_nodes->children.begin(); xml_node != xml_nodes->children.end(); ++xml_node)
		{
			if(xml_node->name != "node")
				continue;

			const k3d::uuid node_class_id = attribute_value<k3d::uuid>(*xml_node, "class", k3d::uuid::null());
			if(node_class_id != k3d::uuid(0x919c3786, 0x619e4e84, 0xb4ad868f, 0x1e77e67c))
				continue;

			element* const xml_properties = find_element(*xml_node, "properties");
			if(!xml_properties)
				continue;

			element* xml_type = 0;
			for(element::elements_t::iterator xml_property = xml_properties->children.begin(); xml_property != xml_properties->children.end(); ++xml_property)
			{
				if(xml_property->name != "property")
					continue;

				if(attribute_text(*xml_property, "name") == "type")
				{
					xml_type = &*xml_property;
					break;
				}
			}

			if(!xml_type)
			{
				k3d::log() << warning << "Upgrading PolySphere node" << std::endl;
				// Add new type, defaults to old behaviour
				xml_properties->append(element("property", "sphereized_cylinder", attribute("name", "type")));
			}
		}
	}

	/// Returns the largest node id in use in the given document
	const k3d::ipersistent_lookup::id_type max_node_id(element& XMLDocument)
	{
		k3d::ipersistent_lookup::id_type result = 0;
		if(element* const xml_nodes = find_element(XMLDocument, "nodes"))
		{
			for(element::elements_t::iterator xml_node = xml_nodes->children.begin(); xml_node != xml_nodes->children.end(); ++xml_node)
			{
				if(xml_node->name != "node")
					continue;

					
				result = std::max(result, attribute_value<k3d::ipersistent_lookup::id_type>(*xml_node, "id", 0));
			}
		}

		return result;
	}
	
	void adjust_dependencies(element& XMLDocument, const k3d::ipersistent_lookup::id_type FromID, const std::string& FromProperty, const k3d::ipersistent_lookup::id_type ToID, const std::string& ToProperty)
	{
		if(element* const xml_dependencies = find_element(XMLDocument, "dependencies"))
		{
			for(element::elements_t::iterator xml_dependency = xml_dependencies->children.begin(); xml_dependency != xml_dependencies->children.end(); ++xml_dependency)
			{
				if(xml_dependency->name != "dependency")
					continue;

				attribute* const to_node = find_attribute(*xml_dependency, "to_node");
				if(!to_node)
					continue;

				if(k3d::from_string<k3d::ipersistent_lookup::id_type>(to_node->value, 0) != ToID)
					continue;

				attribute* const to_property = find_attribute(*xml_dependency, "to_property");
				if(!to_property)
					continue;

				if(to_property->value != ToProperty)
					continue;

				to_node->value = k3d::string_cast(FromID);
				to_property->value = FromProperty;

				break;
			}
		}
	}
	
	/// Upgrades transformable nodes so that their position/orientation/scale properties are replaced with a Transform node
	void upgrade_transformable_nodes(element& XMLDocument)
	{
		element* const xml_nodes = find_element(XMLDocument, "nodes");
		if(!xml_nodes)
			return;

		k3d::ipersistent_lookup::id_type next_node_id = max_node_id(XMLDocument) + 1;

		element::elements_t new_dependencies;
		element::elements_t new_nodes;
		
		for(element::elements_t::iterator xml_node = xml_nodes->children.begin(); xml_node != xml_nodes->children.end(); ++xml_node)
		{
			if(xml_node->name != "node")
				continue;

			const k3d::uuid node_class_id = attribute_value<k3d::uuid>(*xml_node, "class", k3d::uuid::null());
			k3d::iplugin_factory* const node_factory = k3d::plugin(node_class_id);
			if(!node_factory)
				continue;

			const k3d::ipersistent_lookup::id_type node_id = attribute_value<k3d::ipersistent_lookup::id_type>(*xml_node, "id", 0);
			if(!node_id)
				continue;
			
			if(!node_factory->implements(typeid(k3d::itransform_source)))
				continue;

			if(!node_factory->implements(typeid(k3d::itransform_sink)))
				continue;

			element* const xml_properties = find_element(*xml_node, "properties");
			if(!xml_properties)
				continue;
			
			element* xml_position = 0;
			element* xml_orientation = 0;
			element* xml_scale = 0;
			
			for(element::elements_t::iterator xml_property = xml_properties->children.begin(); xml_property != xml_properties->children.end(); ++xml_property)
			{
				if(xml_property->name != "property")
					continue;

				const std::string property_name = attribute_text(*xml_property, "name");
				if(property_name == "position")
					xml_position = &(*xml_property);
				else if(property_name == "orientation")
					xml_orientation = &(*xml_property);
				else if(property_name == "scale")
					xml_scale = &(*xml_property);
			}

			if(!(xml_position && xml_orientation && xml_scale))
				continue;
			
			if(xml_position->text == "0 0 0" && xml_orientation->text == "0 0 0 1" && xml_scale->text == "1 1 1")
				continue;

			const std::string node_name = attribute_text(*xml_node, "name");
			k3d::log() << warning << "Upgrading old transformable node " << node_name << std::endl;

			const k3d::vector3 position = k3d::from_string<k3d::vector3>(xml_position->text, k3d::vector3(0, 0, 0));
			const k3d::angle_axis orientation = k3d::from_string<k3d::angle_axis>(xml_orientation->text, k3d::angle_axis(0, k3d::vector3(0, 0, 1)));
			const k3d::vector3 scale = k3d::from_string<k3d::vector3>(xml_scale->text, k3d::vector3(1, 1, 1));
			const k3d::matrix4 matrix = k3d::translation3D(position) * k3d::rotation3D(orientation) * k3d::scaling3D(scale);

			adjust_dependencies(XMLDocument, next_node_id, "input_matrix", node_id, "input_matrix");
			
			new_dependencies.push_back(
				element("dependency",
					attribute("from_node", next_node_id),
					attribute("from_property", "output_matrix"),
					attribute("to_node", node_id),
					attribute("to_property", "input_matrix")));
			
			new_nodes.push_back(
				element("node",
					attribute("name", "Transformation"),
					attribute("class", k3d::classes::FrozenTransformation()),
					attribute("id", next_node_id),
					element("properties",
						element("property", k3d::string_cast(k3d::identity3D()),
							attribute("name", "input_matrix")),
						element("property", k3d::string_cast(matrix),
							attribute("name", "matrix")))));

			++next_node_id;
		}

		xml_nodes->children.insert(xml_nodes->children.end(), new_nodes.begin(), new_nodes.end());

		element& xml_dependencies = XMLDocument.safe_element("dependencies");
		xml_dependencies.children.insert(xml_dependencies.children.end(), new_dependencies.begin(), new_dependencies.end());
	}
	
	/// Modifies XML as-needed so that both legacy and recent documents can be loaded with the same code
	void upgrade_document(element& XMLDocument)
	{
		upgrade_document_schema(XMLDocument);
		upgrade_poly_sphere_nodes(XMLDocument);
		upgrade_transformable_nodes(XMLDocument);
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::application_plugin<document_reader>, k3d::interface_list<k3d::idocument_read_format> > factory(
			k3d::classes::DocumentReader(),
			"DocumentReader",
			_("K-3D Native ( .k3d )"),
			"");

		return factory;
	}
};

k3d::iplugin_factory& document_reader_factory()
{
	return document_reader::get_factory();
}

} // namespace libk3dcore

