// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/auto_ptr.h>
#include <k3dsdk/bitmap_source.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/file_filter.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/ibitmap_read_format.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/options.h>
#include <k3dsdk/persistent.h>

using namespace k3d::xml;

#include <boost/filesystem/operations.hpp>

#include <iterator>

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_file_in

class bitmap_file_in :
	public k3d::bitmap_source<k3d::persistent<k3d::node> >
{
	typedef k3d::bitmap_source<k3d::persistent<k3d::node> > base;

public:
	bitmap_file_in(k3d::idocument& Document) :
		base(Document),
		m_file(init_owner(*this) + init_name("file") + init_label(_("File")) + init_description(_("Browse for an input bitmap")) + init_value(boost::filesystem::path()) + init_path_mode(k3d::ipath_property::READ) + init_path_type(k3d::options::path::bitmaps()))
	{
		m_file.changed_signal().connect(sigc::mem_fun(*this, &bitmap_file_in::on_reset_bitmap));
		m_output.need_data_signal().connect(sigc::mem_fun(*this, &bitmap_file_in::on_create_bitmap));
	}

	void on_reset_bitmap()
	{
		m_output.reset();
	}

	k3d::bitmap* on_create_bitmap()
	{
		std::auto_ptr<k3d::bitmap> bitmap(new k3d::bitmap());

		const boost::filesystem::path file = m_file.value();
		if(!boost::filesystem::exists(file))
			return 0;

		k3d::auto_ptr<k3d::ibitmap_read_format> filter(k3d::auto_file_filter<k3d::ibitmap_read_format>(file));
		return_val_if_fail(filter.get(), 0);

		return_val_if_fail(filter->read_file(file, *bitmap), 0);

		return bitmap.release();
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_file_in>,
			k3d::interface_list<k3d::ibitmap_source> > factory(
				k3d::classes::FileBitmap(),
				"BitmapFileIn",
				_("Loads a bitmap from the filesystem"),
				"Bitmap",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(boost::filesystem::path, immutable_name, change_signal, with_undo, local_storage, no_constraint, path_property, path_serialization) m_file;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_file_in_factory

k3d::iplugin_factory& bitmap_file_in_factory()
{
	return bitmap_file_in::get_factory();
}

} // namespace libk3dbitmap


