// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "color.h"
#include "log.h"
#include "idocument_read_format.h"
#include "idocument_write_format.h"
#include "ifile_format.h"
#include "imesh_source.h"
#include "imesh_sink.h"
#include "itransform_source.h"
#include "itransform_sink.h"
#include "irenderman.h"
#include "iscript_engine.h"
#include "types.h"

#include <map>

namespace k3d
{

namespace detail
{

/// "Wraps" a std::type_info object so it can be used as the key in a sorted container
struct type_info
{
	type_info(const std::type_info& Info) :
		info(Info)
	{
	}

	bool operator<(const type_info& RHS) const
	{
		return info.before(RHS.info);
	}

	const std::type_info& info;
};

/// Defines storage for a mapping of std::type_info objects to their corresponding string representation
typedef std::map<type_info, std::string> type_to_name_map_t;
/// Stores a mapping of std::type_info objects to their corresponding string representation
type_to_name_map_t type_to_name_map;

/// Defines storage for a mapping of string to the corresponding std::type_info
typedef std::map<std::string, type_info> name_to_type_map_t;
/// Stores a mapping of string to the corresponding std::type_info
name_to_type_map_t name_to_type_map;
	
void register_type(const std::type_info& Info, const std::string& Name)
{
	return_if_fail(0 == type_to_name_map.count(type_info(Info)));
	return_if_fail(0 == name_to_type_map.count(Name));
	
	type_to_name_map.insert(std::make_pair(type_info(Info), Name));
	name_to_type_map.insert(std::make_pair(Name, type_info(Info)));
}

void initialize_types()
{
	static bool initialized = false;
	if(initialized)
		return;

	register_type(typeid(bool), "bool");
	register_type(typeid(int), "int");
	register_type(typeid(long), "long");
	register_type(typeid(float), "float");
	register_type(typeid(double), "double");
	register_type(typeid(std::string), "std::string");
	
	register_type(typeid(k3d::color), "k3d::color");
	register_type(typeid(k3d::vector3), "k3d::vector3");
	register_type(typeid(k3d::matrix4), "k3d::matrix4");
	
	register_type(typeid(k3d::ri::point), "k3d::ri::point");
	register_type(typeid(k3d::ri::vector), "k3d::ri::vector");
	register_type(typeid(k3d::ri::normal), "k3d::ri::normal");
	register_type(typeid(k3d::ri::hpoint), "k3d::ri::hpoint");

	register_type(typeid(k3d::idocument_read_format), "k3d::idocument_read_format");
	register_type(typeid(k3d::idocument_write_format), "k3d::idocument_write_format");
	register_type(typeid(k3d::ifile_format), "k3d::ifile_format");
	register_type(typeid(k3d::imesh_source), "k3d::imesh_source");
	register_type(typeid(k3d::imesh_sink), "k3d::imesh_sink");
	register_type(typeid(k3d::itransform_source), "k3d::itransform_source");
	register_type(typeid(k3d::itransform_sink), "k3d::itransform_sink");
	register_type(typeid(k3d::iscript_engine), "k3d::iscript_engine");
	
	initialized = true;
}
	
} // namespace detail

void register_type(const std::type_info& Info, const std::string& Name)
{
	detail::initialize_types();
	detail::register_type(Info, Name);
}

const types_t registered_types()
{
	detail::initialize_types();

	types_t results;
	for(detail::type_to_name_map_t::iterator type = detail::type_to_name_map.begin(); type != detail::type_to_name_map.end(); ++type)
		results.push_back(&type->first.info);
	return results;
}

const std::string type_string(const std::type_info& Info)
{
	detail::initialize_types();

	detail::type_to_name_map_t::iterator type = detail::type_to_name_map.find(detail::type_info(Info));
	if(type != detail::type_to_name_map.end())
		return type->second;

	log() << error << "unknown type: " << Info.name() << std::endl;
	return "";
}

const std::type_info* type_id(const std::string& Name)
{
	detail::initialize_types();

	detail::name_to_type_map_t::iterator type = detail::name_to_type_map.find(Name);
	if(type != detail::name_to_type_map.end())
		return &type->second.info;

	log() << error << "unknown type: " << Name << std::endl;
	return 0;
}

} // namespace k3d

