// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "base64.h"
#include "data.h"
#include "fstream.h"
#include "system.h"

namespace k3d
{

std::ostream& operator<<(std::ostream& Stream, const ipath_property::reference_t RHS)
{
	switch(RHS)
	{
		case ipath_property::ABSOLUTE_REFERENCE:
			Stream << "absolute";
			break;
		case ipath_property::RELATIVE_REFERENCE:
			Stream << "relative";
			break;
		case ipath_property::INLINE_REFERENCE:
			Stream << "inline";
			break;
	}
	
	return Stream;
}

std::istream& operator>>(std::istream& Stream, ipath_property::reference_t& RHS)
{
	std::string temp;
	Stream >> temp;

	if(temp == "absolute")
		RHS = ipath_property::ABSOLUTE_REFERENCE;
	else if(temp == "relative")
		RHS = ipath_property::RELATIVE_REFERENCE;
	else if(temp == "inline")
		RHS = ipath_property::INLINE_REFERENCE;
	else
	{
		log() << error << "unknown enumeration" << std::endl;
		RHS = ipath_property::RELATIVE_REFERENCE;
	}
	
	return Stream;
}

namespace data
{

void save_external_resource(xml::element& Element, const ipersistent::save_context& Context, const std::string& Name, const ipath_property::reference_t Reference, const boost::filesystem::path& Value)
{
	xml::element& xml_storage = Element.append(
	xml::element("property",
	xml::attribute("name", Name),
	xml::attribute("reference", string_cast(Reference))));

	switch(Reference)
	{
		case ipath_property::ABSOLUTE_REFERENCE:
        {
            boost::filesystem::path absolute_path;
            if(Value.is_complete())
            {
                absolute_path = Value;
            }
            else
            {
                absolute_path = Context.root_path / Value;
            }
            xml_storage.append(xml::attribute("absolute_path", absolute_path.native_file_string()));
        }
        break;
		case ipath_property::RELATIVE_REFERENCE:
        {
            boost::filesystem::path relative_path;
            if(Value.is_complete())
            {
                const boost::filesystem::path root_path = Context.root_path;
                const boost::filesystem::path absolute_path(Value);

                boost::filesystem::path::iterator a = root_path.begin();
                boost::filesystem::path::iterator b = absolute_path.begin();
                while(a != root_path.end() && b != absolute_path.end() && *a == *b)
                {
                    ++a;
                    ++b;
                }

                for(; a != root_path.end(); ++a)
                    relative_path /= "..";
                for(; b != absolute_path.end(); ++b)
                    relative_path /= boost::filesystem::path(*b, boost::filesystem::no_check);
            }
            else
            {
                relative_path = Value;
            }
            xml_storage.append(xml::attribute("relative_path", relative_path.string()));
        }
        break;
		case ipath_property::INLINE_REFERENCE:
        {
            boost::filesystem::path relative_path;
            if(Value.is_complete())
            {
                const boost::filesystem::path root_path = Context.root_path;
                const boost::filesystem::path absolute_path(Value);

                boost::filesystem::path::iterator a = root_path.begin();
                boost::filesystem::path::iterator b = absolute_path.begin();
                while(a != root_path.end() && b != absolute_path.end() && *a == *b)
                {
                    ++a;
                    ++b;
                }

                for(; a != root_path.end(); ++a)
                    relative_path /= "..";
                for(; b != absolute_path.end(); ++b)
                    relative_path /= boost::filesystem::path(*b, boost::filesystem::no_check);
            }
            else
            {
                relative_path = Value;
            }
            xml_storage.append(xml::attribute("filename", relative_path.leaf()));
            xml_storage.append(xml::attribute("encoding", "base64"));
            
            k3d::filesystem::ifstream file(Context.root_path / relative_path);
            std::stringstream buffer;
            base64::encode(file, buffer);
            xml_storage.text = "\n" + buffer.str();
        }
        break;
	}
}

void load_external_resource(xml::element& Element, const ipersistent::load_context& Context, ipath_property::reference_t& Reference, boost::filesystem::path& Value)
{
	Reference = xml::attribute_value<ipath_property::reference_t>(Element, "reference", ipath_property::RELATIVE_REFERENCE);
	switch(Reference)
	{
		case ipath_property::ABSOLUTE_REFERENCE:
			try
			{
				Value = boost::filesystem::path(xml::attribute_text(Element, "absolute_path"), boost::filesystem::native);
				log() << info << "Loaded absolute path " << Value.native_file_string() << std::endl;
			}
			catch(std::exception& e)
			{
				log() << error << e.what() << std::endl;
			}
			break;
		case ipath_property::RELATIVE_REFERENCE:
			try
			{
				Value = Context.root_path / boost::filesystem::path(xml::attribute_text(Element, "relative_path"), boost::filesystem::no_check);
				log() << info << "Resolved relative path as " << Value.native_file_string() << std::endl;
			}
			catch(std::exception& e)
			{
				log() << error << e.what() << std::endl;
			}
			break;
		case ipath_property::INLINE_REFERENCE:
            try
            {
            	Value = k3d::system::get_temp_directory() / xml::attribute_text(Element, "filename");
            	k3d::filesystem::ofstream file(Value);
            	std::stringstream buffer(Element.text);
            	base64::decode(buffer, file);

            	log() << info << "Extracted inline document to " << Value.native_file_string() << std::endl;
            }
            catch(std::exception& e)
            {
            	log() << error << e.what() << std::endl;
            }
			break;
	}
}
			
} // namespace data

} // namespace k3d

