// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Provides the k3d-bug-buddy application, which helps users format and send useful bug reports
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/fstream.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/log.h>
#include <k3dsdk/logbufs.h>
#include <k3dsdk/signal_system.h>
#include <k3dsdk/socket.h>
#include <k3dsdk/string_cast.h>
#include <k3dsdk/system.h>
#include <k3dsdk/utility.h>
#include <k3dsdk/version.h>

#include <gtkmm/box.h>
#include <gtkmm/button.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/entry.h>
#include <gtkmm/filechooserdialog.h>
#include <gtkmm/label.h>
#include <gtkmm/main.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/stock.h>
#include <gtkmm/textview.h>
#include <gtkmm/window.h>

#include <boost/date_time/gregorian/gregorian.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/operations.hpp>

#include <iostream>
#include <sstream>
#include <vector>

namespace
{

typedef std::vector<std::string> string_array;

typedef std::vector<std::streambuf*> logbufs_t;

bool g_show_timestamps = true;
bool g_show_process = true;
bool g_syslog = false;
bool g_color_level = false;
k3d::log_level_t g_minimum_log_level = k3d::DEBUG;

/////////////////////////////////////////////////////////////////////////////
// generate_report

/// Generates the text of a bug report by consolidating system data, compile-time and run-time information, plus optional text attachments
const std::string generate_report(const string_array& Attachments)
{
	std::ostringstream buffer;

	// System data ...
	buffer << "Report Generated: " << boost::posix_time::second_clock::universal_time() << " UTC\n";
	buffer << "Package: " << K3D_PACKAGE << "\n";
	buffer << "Version: " << K3D_VERSION << "\n";
	buffer << "Platform: " << K3D_HOST << "\n";
	buffer << "Compiler: " << __VERSION__ << "\n";
	buffer << "Build Time: " << __DATE__ << " " << __TIME__ << " local\n";

	// Attachments ...
	for(string_array::const_iterator attachment = Attachments.begin(); attachment != Attachments.end(); ++attachment)
	{
		const boost::filesystem::path path(*attachment, boost::filesystem::native);
		if(!boost::filesystem::exists(path))
			{
				buffer << "\n\nMissing File Attachment: " << path.native_file_string() << "\n\n";
				continue;
			}

		k3d::filesystem::ifstream file(path);
		buffer << "\n\nFile Attachment: " << path.native_file_string() << "\n\n";
		buffer << file.rdbuf();
	}

	return buffer.str();
}

/////////////////////////////////////////////////////////////////////////////
// generate_smtp_report

/// Generates the final text of a bug report including all user-supplied data
void generate_smtp_report(std::ostream& Stream, const std::string& Contact, const std::string& Summary, const std::string& Detail, const string_array& Attachments)
{
	const std::string boundary = "0x6995453520214d4798bd1aa51991ae73";

	Stream << "To: K-3D Bug Reports\n";
	Stream << "Subject: Automated Bug Report - " << Summary << "\n";
	Stream << "MIME-Version: 1.0\n";
	Stream << "Content-Type: multipart/mixed; boundary=\"" << boundary << "\"\n";
	Stream << "\n";
	Stream << "--" << boundary << "\n";
	Stream << "Content-Type: text/plain; charset=us-ascii\n";
	Stream << "Contact: " << Contact << "\n";
	Stream << "Summary: " << Summary << "\n";
	Stream << Detail << "\n";
	Stream << "Report Generated: " << boost::posix_time::second_clock::universal_time() << " UTC\n";
	Stream << "Package: " << K3D_PACKAGE << "\n";
	Stream << "Version: " << K3D_VERSION << "\n";
	Stream << "Platform: " << K3D_HOST << "\n";
	Stream << "Compiler: " << __VERSION__ << "\n";
	Stream << "Build Time: " << __DATE__ << " " << __TIME__ << " local\n";

	// Attachments ...
	for(string_array::const_iterator attachment = Attachments.begin(); attachment != Attachments.end(); ++attachment)
	{
		const boost::filesystem::path path(*attachment, boost::filesystem::native);
		if(!boost::filesystem::exists(path))
			{
				Stream << "\n\nMissing File Attachment: " << path.native_file_string() << "\n\n";
				continue;
			}

		k3d::filesystem::ifstream file(path);
		Stream << "\n\n";
		Stream << "--" << boundary << "\n";
		Stream << "Content-Type: text/plain; name=\"" << path.leaf() << "\"\n";
		Stream << "Content-Description: " << path.leaf() << "\n";
		Stream << "Content-Disposition: inline; filename=\"" << path.leaf() << "\"\n";
		Stream << "\n";
		Stream << file.rdbuf();
	}
}

/////////////////////////////////////////////////////////////////////////////
// generate_http_report

/// Generates the final text of a bug report including all user-supplied data
void generate_http_report(std::ostream& Stream, const std::string& Contact, const std::string& Summary, const std::string& Detail, const string_array& Attachments, const std::string& boundary)
{
	Stream << "--" << boundary << "\r\n";
	Stream << "Content-Disposition: form-data; name=\"group_id\"\r\n";
	Stream << "\r\n";
	Stream << "11113\r\n";

	Stream << "--" << boundary << "\r\n";
	Stream << "Content-Disposition: form-data; name=\"atid\"\r\n";
	Stream << "\r\n";
	Stream << "111113\r\n";

	Stream << "--" << boundary << "\r\n";
	Stream << "Content-Disposition: form-data; name=\"func\"\r\n";
	Stream << "\r\n";
	Stream << "postadd\r\n";

	Stream << "--" << boundary << "\r\n";
	Stream << "Content-Disposition: form-data; name=\"category_id\"\r\n";
	Stream << "\r\n";
	Stream << "608709\r\n";

	Stream << "--" << boundary << "\r\n";
	Stream << "Content-Disposition: form-data; name=\"artifact_group_id\"\r\n";
	Stream << "\r\n";
	Stream << "405140\r\n";

	Stream << "--" << boundary << "\r\n";
	Stream << "Content-Disposition: form-data; name=\"summary\"\r\n";
	Stream << "\r\n";
	Stream << Summary << "\r\n";

	Stream << "--" << boundary << "\r\n";
	Stream << "Content-Disposition: form-data; name=\"details\"\r\n";
	Stream << "\r\n";
	Stream << "Contact: " << Contact << "\r\n";
	Stream << "Summary: " << Summary << "\r\n";
	Stream << "Report Generated: " << boost::posix_time::second_clock::universal_time() << " UTC\r\n";
	Stream << "Package: " << K3D_PACKAGE << "\r\n";
	Stream << "Version: " << K3D_VERSION << "\r\n";
	Stream << "Platform: " << K3D_HOST << "\r\n";
	Stream << "Compiler: " << __VERSION__ << "\r\n";
	Stream << "Build Time: " << __DATE__ << " " << __TIME__ << " local\r\n";
	Stream << Detail << "\r\n";

	for(string_array::const_iterator attachment = Attachments.begin(); attachment != Attachments.end(); ++attachment)
	{
		const boost::filesystem::path path(*attachment, boost::filesystem::native);
		if(!boost::filesystem::exists(path))
		{
			k3d::log() << error << "Missing File Attachment: " << path.native_file_string() << std::endl;
			continue;
		}

		Stream << "--" << boundary << "\r\n";
		Stream << "Content-Disposition: form-data; name=\"add_file\"\r\n";
		Stream << "\r\n";
		Stream << "1\r\n";

		k3d::filesystem::ifstream file(path);
		Stream << "--" << boundary << "\r\n";
		Stream << "Content-Disposition: form-data; name=\"input_file\"; filename=\"" << path.leaf() << "\"\r\n";
		Stream << "Content-Type: application/octet-stream\r\n";
		Stream << "\r\n";
		Stream << file.rdbuf();

		Stream << "--" << boundary << "\r\n";
		Stream << "Content-Disposition: form-data; name=\"file_description\"\r\n";
		Stream << "\r\n";
		Stream << "Blackbox Recorder Log\r\n";

		// SF only supports one attachment at-a-time
		break;
	}

	Stream << "--" << boundary << "\r\n";
	Stream << "Content-Disposition: form-data; name=\"submit\"\r\n";
	Stream << "\r\n";
	Stream << "SUBMIT\r\n";
}

/////////////////////////////////////////////////////////////////////////////
// main_window

class main_window :
	public Gtk::Window
{
public:
	main_window(const string_array& Attachments) :
		m_attachments(Attachments),
		m_save(Gtk::Stock::SAVE),
		m_cancel(Gtk::Stock::CANCEL),
		m_send(_("Send")),
		m_privacy_concerns(_("privacy concerns"))
	{
		set_title(_("K-3D Bug Buddy"));
		set_border_width(6);

		Gtk::Label* const title = new Gtk::Label();
		title->set_markup(_("<span size=\"xx-large\" weight=\"heavy\">Welcome to K-3D Bug Buddy</span>"));

		Gtk::Label* const warning = new Gtk::Label(
			_("It appears that a previous K-3D session may have terminated unexpectedly.\nK-3D Bug Buddy will help you generate a bug report based on your Black Box Recorder log."));

		Gtk::Label* const summary = new Gtk::Label();
		summary->set_markup(_("<span weight=\"bold\">Required</span> - enter a one-line summary of what happened:"));
		summary->set_alignment(Gtk::ALIGN_LEFT);

		Gtk::Label* const detail = new Gtk::Label();
		detail->set_markup(_("<span weight=\"bold\">Required</span> - describe what happened in detail:"));
		detail->set_alignment(Gtk::ALIGN_LEFT);

		Gtk::ScrolledWindow* const detail_window = new Gtk::ScrolledWindow();
		detail_window->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
		detail_window->add(m_detail);

		Gtk::Label* const report = new Gtk::Label();
		report->set_markup(_("<span weight=\"bold\">Automated crash data</span> - generated by your Black Box Recorder:"));
		report->set_alignment(Gtk::ALIGN_LEFT);
		Gtk::HButtonBox* const disclaimer_buttons = new Gtk::HButtonBox(Gtk::BUTTONBOX_END);
		disclaimer_buttons->pack_start(m_privacy_concerns, Gtk::PACK_SHRINK);

		Gtk::ScrolledWindow* const report_window = new Gtk::ScrolledWindow();
		report_window->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
		report_window->add(m_report);
		report_window->get_default_style()->set_bg(Gtk::STATE_NORMAL, Gdk::Color("blue"));

		Gtk::Label* const send_label = new Gtk::Label();
		send_label->set_markup(_("<span weight=\"bold\">File an automatic bug report</span> - enter your contact <span weight=\"bold\">e-mail</span> address:"));
		send_label->set_alignment(Gtk::ALIGN_LEFT);

		Gtk::HButtonBox* const send_buttons = new Gtk::HButtonBox(Gtk::BUTTONBOX_END);
		send_buttons->pack_start(m_send, Gtk::PACK_SHRINK);

		Gtk::Label* const save_label = new Gtk::Label();
		save_label->set_markup(_("Or, save the report and send it to the K-3D developers:"));
		save_label->set_alignment(Gtk::ALIGN_LEFT);

		Gtk::HButtonBox* const save_buttons = new Gtk::HButtonBox(Gtk::BUTTONBOX_END);
		save_buttons->pack_start(m_cancel, Gtk::PACK_SHRINK);
		save_buttons->pack_start(m_save, Gtk::PACK_SHRINK);

		Gtk::VBox* const vbox = new Gtk::VBox(false, 10);
		vbox->pack_start(*Gtk::manage(title), Gtk::PACK_SHRINK);
		vbox->pack_start(*Gtk::manage(warning), Gtk::PACK_SHRINK);
		vbox->pack_start(*Gtk::manage(summary), Gtk::PACK_SHRINK);
		vbox->pack_start(m_summary, Gtk::PACK_SHRINK);
		vbox->pack_start(*Gtk::manage(detail), Gtk::PACK_SHRINK);
		vbox->pack_start(*Gtk::manage(detail_window), Gtk::PACK_EXPAND_WIDGET);
		Gtk::HBox* const report_hbox = new Gtk::HBox(false, 10);
		report_hbox->pack_start(*Gtk::manage(report), Gtk::PACK_SHRINK);
		report_hbox->pack_start(*Gtk::manage(disclaimer_buttons), Gtk::PACK_SHRINK);
		vbox->pack_start(*Gtk::manage(report_hbox), Gtk::PACK_SHRINK);
		vbox->pack_start(*Gtk::manage(report_window), Gtk::PACK_EXPAND_WIDGET);
		vbox->pack_start(*Gtk::manage(send_label), Gtk::PACK_SHRINK);
		vbox->pack_start(m_contact, Gtk::PACK_SHRINK);
		vbox->pack_start(*Gtk::manage(send_buttons), Gtk::PACK_SHRINK);
		vbox->pack_start(*Gtk::manage(save_label), Gtk::PACK_SHRINK);
		vbox->pack_start(*Gtk::manage(save_buttons), Gtk::PACK_SHRINK);

		add(*Gtk::manage(vbox));

		m_detail.get_buffer()->set_text("Description of problem:\n\nSteps to reproduce problem:\n\n1. \n2. \n3. \n\nExpected results:\n\nActual results:\n\nHow often does this happen?\n\nAdditional information:\n");

		m_report.get_buffer()->set_text(generate_report(m_attachments));
		m_report.set_editable(false);

		m_save.signal_clicked().connect(sigc::mem_fun(*this, &main_window::on_save));
		m_cancel.signal_clicked().connect(sigc::mem_fun(*this, &main_window::on_cancel));
		m_send.signal_clicked().connect(sigc::mem_fun(*this, &main_window::on_send_http));
		m_privacy_concerns.signal_clicked().connect(sigc::mem_fun(*this, &main_window::on_privacy_concerns));

		set_position(Gtk::WIN_POS_CENTER);
		show_all();
	}

private:
	void on_cancel()
	{
		Gtk::MessageDialog nag(*this, _("Are you sure?  The most reliable way to get a bug fixed is to report it ..."), false, Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_OK_CANCEL, true);
		nag.set_default_response(Gtk::RESPONSE_CANCEL);

		if(Gtk::RESPONSE_OK == nag.run())
			hide();
	}

	bool ready_to_go()
	{
		if(!m_summary.get_text().size())
		{
			Gtk::MessageDialog nag(*this, _("Please include a summary for this bug report"), false, Gtk::MESSAGE_INFO, Gtk::BUTTONS_OK, true);
			nag.run();

			return false;
		}

		if(!m_contact.get_text().size())
		{
			Gtk::MessageDialog nag(*this, _("Please include a contact email address"), false, Gtk::MESSAGE_INFO, Gtk::BUTTONS_OK, true);
			nag.run();

			return false;
		}

		return true;
	}

	void on_save()
	{
		if(!ready_to_go())
			return;

		Gtk::FileChooserDialog chooser(*this, _("Save bug report:"), Gtk::FILE_CHOOSER_ACTION_SAVE);
		chooser.set_filename("k3d.bug");
		chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
		chooser.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_OK);

		if(Gtk::RESPONSE_OK == chooser.run())
		{
			const boost::filesystem::path path(chooser.get_filename(), boost::filesystem::native);
			k3d::filesystem::ofstream file(path);
			generate_smtp_report(
				file,
				m_contact.get_text(),
				m_summary.get_text(),
				m_detail.get_buffer()->get_text(),
				m_attachments);

			const std::string message = k3d::string_cast(boost::format(_("Bug report saved as %1% .  Don't forget to e-mail it to the K-3D Developers")));
			Gtk::MessageDialog(*this, message, false, Gtk::MESSAGE_INFO, Gtk::BUTTONS_OK, true).run();

			hide();
		}
	}

	void on_send_http()
	{
		if(!ready_to_go())
			return;

		const std::string boundary = "0x6995453520214d4798bd1aa51991ae73";

		std::stringstream content;
		generate_http_report(content, m_contact.get_text(), m_summary.get_text(), m_detail.get_buffer()->get_text(), m_attachments, boundary);

		std::stringstream header;
		header << "POST /tracker/?func=add&group_id=11113&atid=111113 HTTP/1.1\r\n";
		header << "Host: sourceforge.net\r\n";
		header << "User-Agent: k3d-bug-buddy\r\n";
		header << "Connection: close\r\n";
		header << "Content-Type: multipart/form-data; boundary=" << boundary << "\r\n";
		header << "Content-Length: " << content.str().size() << "\r\n";
		header << "\r\n";

		try
		{
//			k3d::socket::endpoint connection = k3d::socket::connect("sourceforge.net", 80);
//			connection.write(header.str());
//			connection.write(content.str());
		}
		catch(k3d::socket::closed& e)
		{
		}
		catch(std::exception& e)
		{
			k3d::log() << error << e.what() << std::endl;

			Gtk::MessageDialog nag(*this, k3d::string_cast(boost::format(_("There was an error sending your report - please save it to a file and send it manually instead.\n\n\"%1%\"")) % e.what()), false, Gtk::MESSAGE_INFO, Gtk::BUTTONS_OK, true);
			nag.run();
			return;
		}

		Gtk::MessageDialog nag(*this, _("Thanks!  Your bug report has been submitted.  You can track your report at:\n\nhttp://sourceforge.net/tracker/?group_id=11113&atid=111113"), false, Gtk::MESSAGE_INFO, Gtk::BUTTONS_OK, true);
		nag.run();

		hide();
	}

	void on_privacy_concerns()
	{
		Gtk::MessageDialog disclaimer(*this,
		_(

"<big><u><b>Disclaimer</b></u></big>\n"
"\n"
"The automated crash data contains the following\n"
"information about your system:\n"
"\n"
"* Report date\n"
"* Application name\n"
"* K-3D version\n"
"* Platform name\n"
"* Compiler version\n"
"* Compilation date\n"
"\n"
"Then follow all the user actions performed since the last application launch.\n"
"It doesn't include any other data such as the scenes, bitmaps or other items loaded during the session.\n"
"\n"
"If you still don't feel comfortable about sending a useful bug report, just hit the <b>Cancel</b> button.\n"

		), true, Gtk::MESSAGE_INFO, Gtk::BUTTONS_OK, true);
		disclaimer.run();
	}

	const string_array m_attachments;

	Gtk::Entry m_summary;
	Gtk::TextView m_detail;
	Gtk::TextView m_report;
	Gtk::Entry m_contact;
	Gtk::Button m_save;
	Gtk::Button m_cancel;
	Gtk::Button m_send;
	Gtk::Button m_privacy_concerns;
};

/////////////////////////////////////////////////////////////////////////////
// usage

/// Prints usage info
void usage(const std::string& Name, std::ostream& Stream)
{
	Stream << "usage: " << Name << " [options] [attachment ...]" << std::endl;
	Stream << std::endl;
	Stream << "  -h, --help               prints this help information and exits" << std::endl;
	Stream << "      --version            prints program version information and exits" << std::endl;
	Stream << std::endl;
}

/////////////////////////////////////////////////////////////////////////////
// version

/// Prints version info
void print_version(std::ostream& Stream)
{
	Stream << "K-3D Bug Buddy Version " << K3D_VERSION << std::endl;
	Stream << "Copyright (c) 1995-2005, Timothy M. Shead.  See the AUTHORS file for contributors." << std::endl;
	Stream << "Licensed by the GNU General Public License.  See the COPYING file for details." << std::endl;
	Stream << "K-3D Home Page: http://www.k-3d.org" << std::endl;
	Stream << std::endl;
}

/////////////////////////////////////////////////////////////////////////////
// setup_logging

/// Sets-up options for logging our output
void setup_logging(const std::string& ProcessName, logbufs_t& LogBufs)
{
	LogBufs.push_back(new k3d::reset_level_buf(k3d::log()));

	if(g_show_timestamps)
		LogBufs.push_back(new k3d::timestamp_buf(k3d::log()));

	if(g_show_process)
		LogBufs.push_back(new k3d::tag_buf("[" + ProcessName + "]", k3d::log()));

	if(g_color_level)
	{
#ifdef K3D_PLATFORM_WIN32
		LogBufs.push_back(new k3d::ansi_to_win32_buf(k3d::log()));
#endif // K3D_PLATFORM_WIN32
		LogBufs.push_back(new k3d::color_level_buf(k3d::log()));
	}

	LogBufs.push_back(new k3d::show_level_buf(k3d::log()));

#ifndef	K3D_PLATFORM_WIN32
	if(g_syslog)
		LogBufs.push_back(new k3d::syslog_buf(k3d::log()));
#endif	// !K3D_PLATFORM_WIN32

	LogBufs.push_back(new k3d::filter_by_level_buf(g_minimum_log_level, k3d::log()));
}

} // namespace

/// Program main
int main(int argc, char* argv[])
{
	const std::string program_name = boost::filesystem::path(argv[0], boost::filesystem::native).leaf();

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, K3D_LOCALE);
	bind_textdomain_codeset(PACKAGE, "UTF-8");
	textdomain(PACKAGE);

	Gtk::Main main(&argc, &argv);

	// Put our command-line arguments in a more useable form ...
	string_array options(&argv[1], &argv[argc]);

	// Print a "help" message ...
	if(std::count(options.begin(), options.end(), "-h") || std::count(options.begin(), options.end(), "--help"))
	{
		usage(program_name, std::cout);
		return 0;
	}

	// Print version data ...
	if(std::count(options.begin(), options.end(), "--version"))
	{
		print_version(std::cout);
		return 0;
	}

	// Setup logging right away ...
	logbufs_t logbufs;
	setup_logging(program_name, logbufs);

	main_window window(options);
	main.run(window);

	// For some reason, cleaning these up causes a segfault
	//std::for_each(logbufs.begin(), logbufs.end(), k3d::delete_object());

	return 0;
}

