// K-3D
// Copyright (c) 2005, Romain Behar
//
// Contact: romainbehar@yahoo.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/classes.h>
#include <k3dsdk/computed_property.h>
#include <k3dsdk/glutility.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/mesh.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// selected_point_number_implementation

class selected_point_number_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	selected_point_number_implementation(k3d::idocument& Document) :
		base(Document),
		m_point_number("point_number", "Point number [number]", k3d::method_call(*this, &selected_point_number_implementation::get_selected_point_number))
	{
		register_property(m_point_number);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &selected_point_number_implementation::on_reset_geometry));

		m_input_mesh.changed_signal().connect(m_point_number.changed_signal().slot());

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &selected_point_number_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	unsigned long get_selected_point_number()
	{
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		for(unsigned long n = 0; n < input->points.size(); ++n)
			if(input->points[n]->selected)
				return n;

		return 0;
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<selected_point_number_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0xa98120b3, 0x35714945, 0xa328b2d9, 0xc1f5092e),
				"SelectedPointNumber",
				"Displays first selected point number in mesh point list",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d::computed_property<unsigned long, k3d::method_call_t<selected_point_number_implementation, unsigned long> > m_point_number;
};

/////////////////////////////////////////////////////////////////////////////
// selected_point_number_factory

k3d::iplugin_factory& selected_point_number_factory()
{
	return selected_point_number_implementation::get_factory();
}

} // namespace libk3dmesh


