from twisted.internet.defer import succeed

from juju.errors import EnvironmentNotFound, ProviderError
from juju.lib.testing import TestCase
from juju.providers.orchestra.machine import OrchestraMachine
from juju.providers.orchestra.tests.common import OrchestraTestMixin


class LaunchMachineTest(TestCase, OrchestraTestMixin):

    def test_bad_data(self):
        d = self.get_provider().start_machine({})
        self.assertFailure(d, ProviderError)

        def verify(error):
            self.assertEquals(
                str(error),
                "Cannot launch a machine without specifying a machine-id")
        d.addCallback(verify)
        return d

    def test_no_zookeeper(self):
        self.setup_mocks()
        self.mock_find_zookeepers()
        self.mocker.replay()

        d = self.get_provider().start_machine({"machine-id": "12345"})
        self.assertFailure(d, EnvironmentNotFound)
        return d

    def test_no_available_machines(self):
        self.setup_mocks()
        self.mock_find_zookeepers(("jennifer-uid", "jennifer"))
        self.mock_get_systems(acceptable=False, unacceptable=False)
        self.mocker.replay()

        d = self.get_provider().start_machine({"machine-id": "32767"})
        self.assertFailure(d, ProviderError)

        def verify(error):
            self.assertEquals(
                str(error),
                "Could not find any Cobbler systems marked as available and "
                "configured for network boot.")
        d.addCallback(verify)
        return d

    def test_no_acceptable_machines(self):
        self.setup_mocks()
        self.mock_find_zookeepers(("jennifer-uid", "jennifer"))
        self.mock_get_systems(acceptable=False)
        self.mocker.replay()

        d = self.get_provider().start_machine({"machine-id": "32767"})
        self.assertFailure(d, ProviderError)

        def verify(error):
            self.assertEquals(
                str(error),
                "All available Cobbler systems were also marked as acquired "
                "(instances: bad-system-uid).")
        d.addCallback(verify)
        return d

    def test_actually_launch(self):
        self.setup_mocks()
        self.mock_find_zookeepers(("jennifer-uid", "jennifer"))
        self.mock_get_systems()
        self.mock_acquire_system()
        self.mock_set_ks_meta(
            self.get_verify_ks_meta(42, "launch_user_data"))
        self.mock_start_system()
        self.mock_describe_systems(succeed([{
            "uid": "winston-uid",
            "name": "winston",
            "mgmt_classes": ["acquired"]}]))
        self.mocker.replay()

        def verify_machines(machines):
            (machine,) = machines
            self.assertTrue(isinstance(machine, OrchestraMachine))
            self.assertEquals(machine.instance_id, "winston-uid")
            self.assertEquals(machine.dns_name, "winston")
        d = self.get_provider().start_machine({"machine-id": "42"})
        d.addCallback(verify_machines)
        return d
