from xmlrpclib import Fault
from yaml import dump

from twisted.internet.defer import succeed

from juju.errors import ProviderError
from juju.lib.testing import TestCase
from juju.providers.orchestra.machine import OrchestraMachine
from juju.providers.orchestra.tests.common import OrchestraTestMixin


class OrchestraBootstrapTest(TestCase, OrchestraTestMixin):

    def mock_verify(self):
        self.getPage("http://somewhe.re/webdav/bootstrap-verify",
                     method="PUT", postdata="storage is writable")
        self.mocker.result(succeed(True))

    def mock_save_state(self):
        data = dump({"zookeeper-instances": ["winston-uid"]})
        self.getPage("http://somewhe.re/webdav/provider-state",
                     method="PUT", postdata=data)
        self.mocker.result(succeed(None))

    def test_already_bootstrapped(self):
        self.setup_mocks()
        self.mock_find_zookeepers(("winston-uid", "winston"))
        self.mocker.replay()

        def verify_machines(machines):
            (machine,) = machines
            self.assertTrue(isinstance(machine, OrchestraMachine))
            self.assertEquals(machine.instance_id, "winston-uid")
        d = self.get_provider().bootstrap()
        d.addCallback(verify_machines)
        return d

    def test_no_machines_available(self):
        self.setup_mocks()
        self.mock_find_zookeepers()
        self.mock_verify()
        self.mock_get_systems(acceptable=False)
        self.mocker.replay()
        d = self.get_provider().bootstrap()
        self.assertFailure(d, ProviderError)

    def verify_auth_error(self, error):
        self.setup_mocks()
        self.mock_find_zookeepers()
        self.mock_verify()
        self.mock_get_systems()
        self.mock_acquire_system(error)
        self.mocker.replay()
        d = self.get_provider().bootstrap()
        self.assertFailure(d, type(error))

    def test_non_auth_fault(self):
        return self.verify_auth_error(Fault("blah", "some random error"))

    def test_non_auth_error(self):
        return self.verify_auth_error(Exception("fiddlesticks"))

    def verify_change_failures(self, **kwargs):
        self.setup_mocks()
        self.mock_find_zookeepers()
        self.mock_verify()
        self.mock_get_systems()
        self.mock_acquire_system()
        self.mock_set_ks_meta(
            self.get_verify_ks_meta(0, "bootstrap_user_data"), **kwargs)
        self.mocker.replay()
        d = self.get_provider().bootstrap()
        self.assertFailure(d, ProviderError)

    def test_cannot_modify_machine(self):
        self.verify_change_failures(fail_modify=True)

    def test_cannot_save_machine(self):
        self.verify_change_failures(fail_save=True)

    def test_launch_available_machine(self):
        self.setup_mocks()
        self.mock_find_zookeepers()
        self.mock_verify()
        self.mock_get_systems()
        self.mock_acquire_system()
        self.mock_set_ks_meta(
            self.get_verify_ks_meta(0, "bootstrap_user_data"))
        self.mock_start_system()
        self.mock_describe_systems(succeed([{
            "uid": "winston-uid",
            "name": "winston",
            "mgmt_classes": ["acquired"]}]))
        self.mock_save_state()
        self.mocker.replay()

        def verify_machines(machines):
            (machine,) = machines
            self.assertTrue(isinstance(machine, OrchestraMachine))
            self.assertEquals(machine.instance_id, "winston-uid")
            self.assertEquals(machine.dns_name, "winston")
        d = self.get_provider().bootstrap()
        d.addCallback(verify_machines)
        return d
