# This file is part of the Juju Quickstart Plugin, which lets users set up a
# Juju environment in very few steps (https://launchpad.net/juju-quickstart).
# Copyright (C) 2013 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it under
# the terms of the GNU Affero General Public License version 3, as published by
# the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranties of MERCHANTABILITY,
# SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Juju Quickstart field definitions.

A field is a simple object describing a value, e.g. a label or a help text to
be associated to that value. A field also provides the logic to display,
validate and normalize the given data.

This module is useful as part of the environments metadata definition, in which
each provider type is associated with a sequence of fields. Those fields
describe how an environment of that type should look like, and provide a way to
validate the whole environment on a per-field basis.

See quickstart.models.envs.get_env_type_db for an example of how this works.
"""

from __future__ import unicode_literals

import os
import uuid


class Field(object):
    """Describe a piece of information.

    Also provide the logic to display, normalize and validate input data.

    Field subclasses can define a "field_type" class attribute that can be
    used by view code to choose an appropriate widget to use for that type of
    field instances, e.g. a "bool" field type indicates that a checkbox is
    appropriate when editing that field value. The following field types are
    defined by fields in this module:
        - bool: as mentioned, values are expected to be boolean values;
    If field_type is not specified, view code assumes the values can be
    edited using the usual input edit widget which handles multi-line strings.

    Field instances have the following attributes:
        - name: the key identifying a specific piece of information. In the
          environments context this can be, for instance, "admin-secret" or
          "default-series";
        - label: a human friendly string identifying this field
          (e.g. "Admin Secret");
        - help: help text associated with this field
          (e.g. "the password you use for authenticating");
        - default: the default value if the value is not set (None). This is
          used only in the validation process, and can be used by view code
          to display the default value for a field;
        - required: True if this is a required field, False otherwise;
        - readonly: True if the associated value must be considered immutable.

    Field instances also expose the following methods:
        - display(value): how the value should be displayed by views
          (usually just the value itself as a unicode string is returned);
        - normalize(value): return the normalized value, e.g. a string field
          might return a stripped version of the input value. Returning None
          means the value for that field is unset;
        - validate(value): validate the given value, raising a ValueError if
          the input value is not valid, returning None otherwise;
        - generate(): this optional method indicates the value associated with
          this field can be optionally automatically generated by view code.
          When implemented, this method must return a suitable generated value.

    Note that it is not safe to call normalize on a value if that value has not
    been previously validated.
    """

    # Since this is the default field the type is not specified.
    field_type = None

    def __init__(
            self, name, label=None, help='', default=None, required=False,
            readonly=False):
        """Initialize a field. Only the name identifier is required."""
        self.name = name
        self.label = name if label is None else label
        self.help = help
        self.default = default
        self.required = required
        self.readonly = readonly

    def __repr__(self):
        name = self.name.encode('utf-8')
        return b'<{}: {}>'.format(self.__class__.__name__, name)

    def display(self, value):
        """Return a value to display.

        Override this method to change how the value is displayed in view code.
        """
        return unicode(value)

    def normalize(self, value):
        """Return a normalized version of the given value."""
        return value

    def validate(self, value):
        """Validate the given value.

        Raise a ValueError if the given value is required, it is not set and
        no default is provided.
        """
        if self.required and (value is None) and (self.default is None):
            msg = 'a value is required for the {} field'.format(self.label)
            raise ValueError(msg.encode('utf-8'))


class StringField(Field):
    """Values associated with this field must be strings."""

    def normalize(self, value):
        """Strip the string. Return None if the value is not set."""
        if value is None:
            return None
        return value.strip() or None

    def validate(self, value):
        """Check that the value is a string."""
        if not isinstance(value, (unicode, type(None))):
            # Assume view code always works with unicode strings.
            msg = 'the {} field requires a string value'.format(self.label)
            raise ValueError(msg.encode('utf-8'))
        super(StringField, self).validate(self.normalize(value))


class IntField(Field):
    """Values associated with this field must be integers."""

    def __init__(self, name, min_value=None, max_value=None, **kwargs):
        """Initialize an integer field.

        The "min_value" and "max_value" keyword arguments, if provided, are
        used in the validation process.
        """
        super(IntField, self).__init__(name, **kwargs)
        self.min_value = min_value
        self.max_value = max_value

    def normalize(self, value):
        """Return the value as an integer.

        Return None if the value is an empty string or None. In these cases,
        the field value is considered not set.
        """
        if isinstance(value, unicode):
            value = value.strip()
        if value in ('', None):
            return None
        return int(value)

    def validate(self, value):
        """Validate the integer value.

        Raise a ValueError if:
            - the normalized value is None but the value is required;
            - the normalized field is set but it is not an integer number;
            - the normalized field is a number but not in the range defined
              by self.min_value and self.max_value.
        """
        label = self.label
        # Ensure the value, if set, is an integer.
        msg = 'the {} field requires an integer value'.format(label)
        # Avoid implicit boolean to integer conversion.
        if isinstance(value, bool):
            raise ValueError(msg.encode('utf-8'))
        try:
            value = self.normalize(value)
        except (TypeError, ValueError):
            raise ValueError(msg.encode('utf-8'))
        # Ensure the value is set if required.
        super(IntField, self).validate(value)
        if value is None:
            return
        # Ensure the value is in the given range.
        min_value = self.min_value
        max_value = self.max_value
        if (min_value is not None) and (max_value is not None):
            if not (min_value <= value <= max_value):
                msg = 'the {} value must be in the {}-{} range'.format(
                    label, min_value, max_value)
                raise ValueError(msg.encode('utf-8'))
        elif min_value is not None:
            if value < min_value:
                msg = 'the {} value must be >= {}'.format(label, min_value)
                raise ValueError(msg.encode('utf-8'))
        elif max_value is not None:
            if value > max_value:
                msg = 'the {} value must be <= {}'.format(label, max_value)
                raise ValueError(msg.encode('utf-8'))


class BoolField(Field):
    """Values associated with this field must be booleans."""

    field_type = 'bool'

    def __init__(self, name, allow_mixed=True, **kwargs):
        """Add the allow_mixed keyword argument.

        By default allow_mixed is True, and that means the field can be in a
        "not set" state (None). This is relevant for validation and view code.
        """
        super(BoolField, self).__init__(name, **kwargs)
        self.allow_mixed = allow_mixed

    def validate(self, value):
        """Check that the value, if set, is a boolean."""
        types = (bool, type(None)) if self.allow_mixed else (bool,)
        if not isinstance(value, types):
            msg = 'the {} field requires a boolean value'.format(self.label)
            raise ValueError(msg.encode('utf-8'))


class UnexpectedField(Field):
    """An unexpected field, used when the value type is unknown."""

    def __init__(self, name, label=None, help=None):
        if help is None:
            help = 'this field is unrecognized and can be safely removed'
        super(UnexpectedField, self).__init__(
            name, label=label, help=help,
            default=None, required=False, readonly=False)

    def normalize(self, value):
        """Try to guess the value type."""
        if isinstance(value, (bool, int, type(None))):
            # For booleans, numbers, and None values, return the value as is.
            return value
        if isinstance(value, unicode):
            # If the value is a string (e.g. because it is returned as a
            # string by view code), try to guess the underlying type.
            value = value.strip()
            # Check if the value is not set and can be discarded.
            if not value:
                return None
            # Check if the value is a number.
            if value.isdigit():
                return int(value)
            # Check if the value is a boolean.
            lower_value = value.lower()
            if lower_value == 'true':
                return True
            if lower_value == 'false':
                return False
            # The value is a string, return it.
            return value
        # If the above did not work, always turn the value into a string.
        return unicode(value)

    def validate(self, value):
        """Unexpected values are always valid."""
        pass


class ChoiceField(StringField):
    """A string field whose value must be included in the given choices."""

    def __init__(self, name, choices=(), **kwargs):
        """Initialize the choices field with the given choices."""
        super(ChoiceField, self).__init__(name, **kwargs)
        self.choices = tuple(choices)

    def validate(self, value):
        """Check the field is set if required.

        If the field is set, also check it is included in self.choices.
        """
        # The parent field ensures the value is set if required.
        super(ChoiceField, self).validate(value)
        value = self.normalize(value)
        choices = list(self.choices)
        # If the field is not required, or if a default value is set, None is
        # added to the list of valid choices, so that a "not set" value is
        # accepted.
        if (not self.required) or (self.default is not None):
            choices.append(None)
        if value not in choices:
            msg = 'the {} requires the value to be one of the following: {}'
            raise ValueError(
                msg.format(self.label, ', '.join(self.choices)).encode('utf-8')
            )


class FilePathField(StringField):
    """A string field storing a path to an existing file."""

    def validate(self, value):
        """Check the field is set if required.

        If the field is set, also check that the file exists.
        """
        # The parent field ensures the value is set if required.
        super(FilePathField, self).validate(value)
        value = self.normalize(value)
        # If the value is not set, just return without validating the path.
        if value is None:
            return
        if not os.path.isfile(os.path.expanduser(value)):
            raise ValueError('file not found in the specified path')


class SuggestionsStringField(StringField):
    """A string field storing possible value suggestions."""

    def __init__(self, name, suggestions=(), **kwargs):
        """Initialize the choices field with the given choices."""
        super(SuggestionsStringField, self).__init__(name, **kwargs)
        self.suggestions = tuple(suggestions)


class AutoGeneratedStringField(StringField):
    """Can automatically generate string values if they are not provided.

    Subclasses can override the generate method to return customized values.
    """

    def generate(self):
        """Generate a uuid valid value."""
        return '{}-{}'.format(self.name[:3], uuid.uuid4().hex)


class PasswordField(StringField):
    """Assume values associated with this field represent sensible data."""

    def display(self, value):
        """Obfuscate the value."""
        if value:
            return '*****'
        return 'None'


class AutoGeneratedPasswordField(AutoGeneratedStringField, PasswordField):
    """Values are passwords which can be automatically generated."""
