// Copyright 2012 The Go Authors.  All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// +build darwin freebsd linux netbsd openbsd

package ipv4_test

import (
	"code.google.com/p/go.net/ipv4"
	"net"
	"os"
	"testing"
)

func TestReadWriteMulticastIPPayloadUDP(t *testing.T) {
	if testing.Short() || !*testExternal {
		t.Logf("skipping test to avoid external network")
		return
	}

	c, err := net.ListenPacket("udp4", "224.0.0.0:1024") // see RFC 4727
	if err != nil {
		t.Fatalf("net.ListenPacket failed: %v", err)
	}
	defer c.Close()

	ifi := loopbackInterface()
	if ifi == nil {
		t.Logf("skipping test; an appropriate interface not found")
		return
	}
	dst, err := net.ResolveUDPAddr("udp4", "224.0.0.254:1024") // see RFC 4727
	if err != nil {
		t.Fatalf("net.ResolveUDPAddr failed: %v", err)
	}

	p := ipv4.NewPacketConn(c)
	if err := p.JoinGroup(ifi, dst); err != nil {
		t.Fatalf("ipv4.PacketConn.JoinGroup on %v failed: %v", ifi, err)
	}
	if err := p.SetMulticastInterface(ifi); err != nil {
		t.Fatalf("ipv4.PacketConn.SetMulticastInterface failed: %v", err)
	}
	if err := p.SetMulticastLoopback(true); err != nil {
		t.Fatalf("ipv4.PacketConn.SetMulticastLoopback failed: %v", err)
	}
	runPayloadTransponder(t, p, []byte("HELLO-R-U-THERE"), dst)
}

func TestReadWriteMulticastIPPayloadICMP(t *testing.T) {
	if testing.Short() || !*testExternal {
		t.Logf("skipping test to avoid external network")
		return
	}
	if os.Getuid() != 0 {
		t.Logf("skipping test; must be root")
		return
	}

	c, err := net.ListenPacket("ip4:icmp", "0.0.0.0")
	if err != nil {
		t.Fatalf("net.ListenPacket failed: %v", err)
	}
	defer c.Close()

	ifi := loopbackInterface()
	if ifi == nil {
		t.Logf("skipping test; an appropriate interface not found")
		return
	}
	dst, err := net.ResolveIPAddr("ip4", "224.0.0.254") // see RFC 4727
	if err != nil {
		t.Fatalf("net.ResolveIPAddr failed: %v", err)
	}

	p := ipv4.NewPacketConn(c)
	if err := p.JoinGroup(ifi, dst); err != nil {
		t.Fatalf("ipv4.PacketConn.JoinGroup on %v failed: %v", ifi, err)
	}
	if err := p.SetMulticastInterface(ifi); err != nil {
		t.Fatalf("ipv4.PacketConn.SetMulticastInterface failed: %v", err)
	}
	id := os.Getpid() & 0xffff
	pld := newICMPEchoRequest(id, 1, 128, []byte("HELLO-R-U-THERE"))
	runPayloadTransponder(t, p, pld, dst)
}

func TestReadWriteMulticastIPDatagram(t *testing.T) {
	if testing.Short() || !*testExternal {
		t.Logf("skipping test to avoid external network")
		return
	}
	if os.Getuid() != 0 {
		t.Logf("skipping test; must be root")
		return
	}

	c, err := net.ListenPacket("ip4:icmp", "0.0.0.0")
	if err != nil {
		t.Fatalf("net.ListenPacket failed: %v", err)
	}
	defer c.Close()

	ifi := loopbackInterface()
	if ifi == nil {
		t.Logf("skipping test; an appropriate interface not found")
		return
	}
	dst, err := net.ResolveIPAddr("ip4", "224.0.0.254") // see RFC 4727
	if err != nil {
		t.Fatalf("ResolveIPAddr failed: %v", err)
	}

	r, err := ipv4.NewRawConn(c)
	if err != nil {
		t.Fatalf("ipv4.NewRawConn failed: %v", err)
	}
	if err := r.JoinGroup(ifi, dst); err != nil {
		t.Fatalf("ipv4.RawConn.JoinGroup on %v failed: %v", ifi, err)
	}
	if err := r.SetMulticastInterface(ifi); err != nil {
		t.Fatalf("ipv4.PacketConn.SetMulticastInterface failed: %v", err)
	}
	id := os.Getpid() & 0xffff
	pld := newICMPEchoRequest(id, 1, 128, []byte("HELLO-R-U-THERE"))
	runDatagramTransponder(t, r, pld, nil, dst)
}
