#include <jmp-config.h>

#ifdef HAVE_WIN32COMPAT_H

/* Windows platform compatability layer to implement a POSIX thread like
 *  API for pthread_mutex_xxxxx() and pthread_cond_xxxxx() scheme on top of
 *  windows native threading primitives.
 *  This is not a performance optimized implementation, it does not need
 *  to be since its only used during startup and shutdown of JMP.
 *  
 * There are a few extra libc related functions that are missing from the
 *  WIN32 implementation.
 */

#include <stdio.h>
#include <stdarg.h>
#include <ctype.h>
#include <limits.h>
#ifdef HAVE_WINDOWS_H
 #include <windows.h>
#endif

#include <win32compat.h>

int
my_strcasecmp(const char *a, const char *b)
{
	while(*a && *b) {
		int r;

		r = tolower(*a) - tolower(*b);

		if(r != 0)
			return r;

		a++;
		b++;
	}

	if(*a)
		return *a;

	if(*b)
		return -(*b);

	return 0;
}

int
my_snprintf(char *str, size_t size, const char *format, ...)
{
	int rv;
	va_list ap;
	va_start(ap, format);
	rv = vsprintf(str, format, ap);
	va_end(ap);
	return rv;
}

int
win32_pthread_mutex_destroy(struct win32_pthread_mutex *m)
{
	DeleteCriticalSection(&m->csMutex);
	return 0;
}

int
win32_pthread_mutex_init(struct win32_pthread_mutex *m, struct win32_pthread_mutexattr *a)
{
	InitializeCriticalSection(&m->csMutex);
	return 0;
}

int
win32_pthread_mutex_lock(struct win32_pthread_mutex *m)
{
	EnterCriticalSection(&m->csMutex);
	return 0;
}

#if(_WIN32_WINNT >= 0x0400)
int
win32_pthread_mutex_trylock(struct win32_pthread_mutex *m)
{
	if (TryEnterCriticalSection(&m->csMutex) == TRUE)
		return 0;
	return -1;
}
#endif

int
win32_pthread_mutex_unlock(struct win32_pthread_mutex *m)
{
	LeaveCriticalSection(&m->csMutex);
	return 0;
}


int
win32_pthread_mutexattr_destroy(struct win32_pthread_mutexattr *a)
{
	return 0;
}

int
win32_pthread_mutexattr_init(struct win32_pthread_mutexattr *a)
{
	memset(a, 0, sizeof(*a));
	return 0;
}

int
win32_pthread_mutexattr_settype(struct win32_pthread_mutexattr *a, int t)
{
	a->type = t;
	return 0;
}

int
win32_pthread_mutexattr_gettype(struct win32_pthread_mutexattr *a, int *tp)
{
	*tp = a->type;
	return 0;
}


int
win32_pthread_cond_destroy(struct win32_pthread_cond *v)
{
	DWORD dwWaitResult = WaitForSingleObject(v->hMutex, INFINITE);
	if(dwWaitResult != WAIT_OBJECT_0)
		return -1;
	if(v->waiters > 0) {
		// EBUSY
		return -1;
	}
	CloseHandle(v->hWake);
	v->hWake = NULL;
	CloseHandle(v->hMutex);
	v->hMutex = NULL;
	return 0;
}

int
win32_pthread_cond_init(struct win32_pthread_cond *v, struct win32_pthread_condattr *a)
{
	memset(v, 0, sizeof(*v));

	if((v->hMutex = CreateMutex(NULL, FALSE, NULL)) == NULL)
		goto fail;
	if((v->hWake = CreateEvent(NULL, TRUE, FALSE, NULL)) == NULL)
		goto fail_mutex;
	 
	return 0;

#if 0
fail_event:
	CloseHandle(v->hWake);
#endif
fail_mutex:
	CloseHandle(v->hMutex);
fail:
	return -1;
}

int
win32_pthread_cond_broadcast(struct win32_pthread_cond *v)
{
	DWORD dwWaitResult = WaitForSingleObject(v->hMutex, INFINITE);
	if(dwWaitResult == WAIT_OBJECT_0) {
		// increment generation counter
		v->generation_wakeall++;
		// set event type broadcast
		PulseEvent(v->hWake);
		ReleaseMutex(v->hMutex);
		return 0;
	}
	return -1;
}

int
win32_pthread_cond_signal(struct win32_pthread_cond *v)
{
	DWORD dwWaitResult = WaitForSingleObject(v->hMutex, INFINITE);
	if(dwWaitResult == WAIT_OBJECT_0) {
		// increment generation counter
		v->generation_wakeone++;
		// set event type broadcast
		PulseEvent(v->hWake);
		ReleaseMutex(v->hMutex);
		return 0;
	}
	return -1;
}

#define TIMEB_SEC_MIN   0
#define TIMEB_SEC_MAX   INT_MAX

#define TIMEB_MSEC_VAL  1000
#define TIMEB_MSEC_MIN  0
#define TIMEB_MSEC_MAX  999


#define TIMEB_SANITIZE_MSEC(tb)			\
	(((tb)->millitm > TIMEB_MSEC_MAX) ?	\
	TIMEB_MSEC_MAX :			\
	(((tb)->millitm < TIMEB_MSEC_MIN) ?	\
	TIMEB_MSEC_MIN :			\
	(tb)->millitm))

#define TIMEB_SANITIZE_SEC(tb)			\
	(((tb)->time > TIMEB_SEC_MAX) ?		\
	TIMEB_SEC_MAX :				\
	(((tb)->time < TIMEB_SEC_MIN) ?		\
	TIMEB_SEC_MIN :				\
	(tb)->time))


static struct timeb *
timeb_sub(struct timeb *dest, const struct timeb *a, const struct timeb *b)
{
	struct timeb *rv;
	int msec;
	long sec;

	sec     = 0;
	msec    = TIMEB_SANITIZE_MSEC(a) - TIMEB_SANITIZE_MSEC(b);

	if(msec < TIMEB_MSEC_MIN) {
		sec--;
		msec += TIMEB_MSEC_VAL;
	}

	sec += (TIMEB_SANITIZE_SEC(a) - TIMEB_SANITIZE_SEC(b));

	if(sec < TIMEB_SEC_MIN) {
		dest->time = 0;
		dest->millitm = 0;

		rv = NULL;
	} else {
		dest->time	= sec;
		dest->millitm	= (unsigned short) msec;

		rv = dest;
	}

	return rv;
}

int
win32_pthread_cond_timedwait(struct win32_pthread_cond *v, struct win32_pthread_mutex *m, const struct timeb *w)
{
	int generation_wakeone;
	int generation_wakeall;
	int f_thread_one;
	int rv;

	rv = -1;

	if(v->thread_one_set == 0) {
		/* v->thread_one_set only exists because, I have no MSDN
		 *  documented confirmation that thread ID=0 would never be
		 *  returned by GetCurrentThreadId()
		 */
		v->thread_one_set = 1;
		v->thread_one = GetCurrentThreadId();
		f_thread_one = 1;
	} else {
		f_thread_one = 0;
	}
	generation_wakeone = v->generation_wakeone;
	generation_wakeall = v->generation_wakeall;

	for (;;) {
		DWORD millis;
		DWORD rv_wait;

		if(generation_wakeone != v->generation_wakeone) {
			if(f_thread_one == 0) {
				/* not us this time, try aquire slot, reset */
				if(v->thread_one_set == 0) {
					v->thread_one_set = 1;
					v->thread_one = GetCurrentThreadId();
					f_thread_one = 1;
				}
				generation_wakeone = v->generation_wakeone;
			} else {
				/* wakeup */
				rv = 0;
				break;
			}
		}
		if(generation_wakeall != v->generation_wakeall) {
			/* wakeup */
			rv = 0;
			break;
		}

		if(w != NULL) {
			struct timeb diff;
			struct timeb now;

			ftime (&now);
			timeb_sub (&diff, w, &now);
			if (diff.time <= 0 && diff.millitm <= 0)
				break;

			millis = (diff.time * 1000);
			millis += diff.millitm;

			if(millis <= 0)
				break;
		} else {
			millis = INFINITE;
		}

		v->waiters++;
		win32_pthread_mutex_unlock(m);

		/* CHECKME: Can we loose a wakeup here ? */

		rv_wait = WaitForSingleObject(v->hWake, millis);

		win32_pthread_mutex_lock(m);
		v->waiters--;

		if(rv_wait == WAIT_TIMEOUT) {
			/* ETIMEDOUT */
			break;
		} else if(rv_wait == WAIT_OBJECT_0) {
			/* NOOP - left the generation check above to the business */
		} else /* WAIT_ABANDONED */ {
			break;
		}
	}

	if(f_thread_one) {	/* we own slot */
		v->thread_one_set = 0;
		v->thread_one = 0;
	}

	return rv;
}

int
win32_pthread_cond_wait(struct win32_pthread_cond *v, struct win32_pthread_mutex *m)
{
	int rv;
#if 0
	struct timeb w;
	w.times = TIMEB_SEC_MAX;
	w.millitm = TIMEB_MSEC_MAX;
#endif
	/* Check POSIX spec what does w==NULL mean? */
	rv = win32_pthread_cond_timedwait(v, m, NULL);
	return rv;
}

#endif /* HAVE_WIN32COMPAT_H */

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
