/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */


package org.gnu.gnomevte;

import java.io.File;
import java.io.FileNotFoundException;

/**
 * VteTerminal widget bindings implementation.
 */
public class Terminal extends org.gnu.gtk.Widget {
	
	private int pid_t;
	
	/**
	 * Used in the bindings development.<br>
	 * You should not use this constructor.<br>
	 * Use <code>Terminal ()</code> or <code>Terminal (command, args, workingDir)</code> instead.
	 */
	protected Terminal (int handle){
		super (handle);
	}

	/**
	 * Creates a new Termina widget.
	 * This constructor launchs a shell inside de widget.
	 */
	public Terminal (){
		super (vte_terminal_new ());
	}

	/**
	 * Creates a new Terminal widget and executes <code>command</code> inside it.
	 *
	 * @param command Command launched inside the terminal.
	 * @param args arguments passed to the <code>command</code>.
	 * @param workingDir working directory passed to the process.
	 */
	public Terminal (String command, String args[], String workingDir){
		this();
		pid_t = forkCommand (command, args, workingDir, true, true, true);
	}

	/**
	 * Creates an new Terminal widget and launches a shell inside it.
	 *
	 * @return The widget with a shell in it.
	 */
	public static Terminal terminalAndShell (){
		Terminal t = new Terminal (null, null, null);
		return t;
	}
		
	/**
	 * Forks the <code>command</code> and show it inside the widget.
	 * 
	 * @param command program to launch in a new process.
	 * @param args The argument list available to the executed program.
	 * @param directory working directory available to the executed program.
	 * FIXME: investigate the meaning of laslog, utmp, wtmp. 
	 * @return Integer representing the process id.
	 */
	public int forkCommand (String command, String args[], String directory, boolean lastlog, boolean utmp, boolean wtmp){
		byte[] _command = null;
		byte[] _directory = null;
		if (command != null)
			_command = command.getBytes ();
		if (directory != null)
			_directory = directory.getBytes ();
		return vte_terminal_fork_command (getHandle(), _command, args, _directory, lastlog, utmp, wtmp);
		
	}

	/** 
	 * Send data to the terminal to display to handle in some way.<br> 
	 * If data is null, it will be ignored.
	 * 
	 * @param data Sent to the terminal.
	 */
	public void feed (String data){
		if (data != null)
			vte_terminal_feed (getHandle(), data.getBytes (), data.length ());
	}
	
	/**	
	 * Send data to the terminal's forked command to handle in some way.<br>
	 * If data is null, it will be ignored.
	 * 
	 * @param data Sent to the terminal.
	 */
	public void feedChild (String data){
		if (data != null){
			vte_terminal_feed_child (getHandle(), data.getBytes (), data.length ());
		}
	}

	/**
	 * Copy currently selected text to the clipboard.
	 */
	public void copyClipboard (){
		vte_terminal_copy_clipboard (getHandle());
	}

	/**
	 * Paste clipboard text from the clipboard to the terminal.
	 */
	public void pasteClipboard (){
		vte_terminal_paste_clipboard (getHandle());
	}

	/*
	 * TODO: Implement?
	 *
	public void copyPrimary (){
	}

	/**
	 * TODO: Implement ?
	 *
	public void pastePrimary (){
	}*/

	/**
	 * Set the terminal's size.
	 *
	 * @param colums The terminal's width.
	 * @param rows The terminal's height.
	 */
	public void setSize (int columns, int rows){
		vte_terminal_set_size (getHandle(), columns, rows);
	}
	
	/**
	 * Set the terminal's audible bell.
	 *
	 * @param audible <br> If true, the terminal emits a Bip to attract users attention.
	 */
	public void setAudibleBell (boolean audible){
		vte_terminal_set_audible_bell (getHandle(), audible);
	}

	/**
	 *  Get the terminal's audible bell state.
	 *
	 *  @return The audible bell state.
	 */
	public boolean isBellAudible (){
		return vte_terminal_get_audible_bell (getHandle());
	}

	/**
	 * Set the terminal's visible bell state.
	 * 
	 * @param visible <br> If true, the terminal blinks to attract users attention.
	 */
	public void setVisibleBell (boolean visible){
		vte_terminal_set_visible_bell (getHandle(), visible);
	}

	/**
	 *  Get the terminal's audible bell state.
	 *
	 *  @return The visual bell state
	 */
	public boolean isBellVisible (){
		return vte_terminal_get_visible_bell (getHandle());
	}

	/**
	 * If true, scrolls the widget down following the output.
	 *
	 * @param scrollOnOutput If true, scroll.
	 */
	public void setScrollOnOutput (boolean scrollOnOutput){
		vte_terminal_set_scroll_on_output (getHandle(), scrollOnOutput);
	}

	/**
	 * If true, scrolls the widget down when pressing following the keyboard press.
	 *
	 * @param scrollOnKeystroke If true, scroll.
	 */
	public void setScrollOnKeystroke (boolean scrollOnKeystroke){
		vte_terminal_set_scroll_on_keystroke (getHandle(), scrollOnKeystroke);
	}

	/**
	 * Restores the default colors.
	 * 
	 */
	public void setDefaultColors() {
		vte_terminal_set_default_colors(getHandle());
	}
	 
	//setBackgroudImage(Pixbuf);
	
	/**
	 * Set the <code>file</code> as the background image of the terminal.
	 *
	 * @param file File to set as background.
	 * 
	 * @throws FileNotFoundException if the image doesn't exist.
	 */
	public void setBackgroudImage(String file) throws FileNotFoundException{
		File f = new File(file);
		if (f.exists())
			vte_terminal_set_background_image_file(getHandle(), file.getBytes());
		else
			throw new FileNotFoundException();
	}
		
	/**
	 * Set whether or not the cursor blinks.
	 *
	 * @param blinks If true, blinks.
	 */
	public void setCursorBlinks(boolean blinks){
		vte_terminal_set_cursor_blinks(getHandle(), blinks);
	}
	
	/**
	 * Set the number of scrollback lines.
	 *
	 * @param lines The number of lines to save in the buffer.
	 */
	public void setScrollbackLines(int lines){
		vte_terminal_set_scrollback_lines(getHandle(), lines);
	}
		
	/**
	 * Sets the terminal's background saturation level.
	 *
	 * @param saturation The saturation level.
	 */
	public void setBackgroundSaturation (int saturation){
		vte_terminal_set_background_saturation (getHandle(), saturation);
	}

	/**
	 * Sets the terminal backgroud transparent or not.
	 *
	 * @param transparent Transparent if <code>true</code>.
	 */
	public void setBackgroundTransparent (boolean transparent){
		vte_terminal_set_background_transparent (getHandle(), transparent);
	}
		
		

	static {
		System.loadLibrary ("VTEGNOME");
	}

	/*-------- NATIVE METHODS -----------*/
	
	static native int vte_terminal_new();

	
	
	static native int vte_terminal_fork_command(int handle,
				byte[] command, String[] argv,
				byte[] directory,
				boolean lastlog,
				boolean utmp,
				boolean wtmp);

	/* Send data to the terminal to display, or to the terminal's forked command
	to handle in some way.  If it's 'cat', they should be the same. */
	static native void vte_terminal_feed(int handle, byte[] data, int length);
	static native void vte_terminal_feed_child (int handle,
			     byte[] data, long length);

	/* Copy currently-selected text to the clipboard, or from the clipboard to
 	* the terminal. */
	static native void vte_terminal_copy_clipboard (int handle);
	static native void vte_terminal_paste_clipboard (int handle);
	static native void vte_terminal_copy_primary(int handle);
	static native void vte_terminal_paste_primary(int handle);

	/* Set the terminal's size. */
	static native void vte_terminal_set_size(int handle, int columns, int rows);

	/* Set various one-off settings. */
	static native void vte_terminal_set_audible_bell(int handle, boolean is_audible);
	static native boolean vte_terminal_get_audible_bell(int handle);
	static native void vte_terminal_set_visible_bell(int handle, boolean is_visible);
	static native boolean vte_terminal_get_visible_bell(int handle);
	static native void vte_terminal_set_scroll_on_output(int handle, boolean scroll);
	static native void vte_terminal_set_scroll_on_keystroke(int handle, boolean scroll);

	/* Set the color scheme. */
	static native void vte_terminal_set_color_dim(int handle, int gdkColor);
	static native void vte_terminal_set_color_bold(int handle, int gdkColor);
	static native void vte_terminal_set_color_foreground(int handle, int gdkColor);
	static native void vte_terminal_set_color_background(int handle, int gdkColor);
	
	// foreground, background and palette are gdk.Color
	static native void vte_terminal_set_colors(int handle, int foreground, int background, int palette, int palette_size);
	static native void vte_terminal_set_default_colors(int handle);

	/* Background effects. */

	//image is a GdkPixbuf
	static native void vte_terminal_set_background_image(int handle, int image);
	static native void vte_terminal_set_background_image_file(int handle, byte[] path);
	static native void vte_terminal_set_background_saturation(int handle, double saturation);
	static native void vte_terminal_set_background_transparent(int handle, boolean transparent);

	/* Set whether or not the cursor blinks. */
	static native void vte_terminal_set_cursor_blinks(int handle, boolean blink);

	/* Set the number of scrollback lines, above or at an internal minimum. */
	static native void vte_terminal_set_scrollback_lines(int handle, int lines);

	/* Append the input method menu items to a given shell. */
	// menushell: GtkMenuShell
	static native void vte_terminal_im_append_menuitems(int handle, int menushell);

	/* Set or retrieve the current font. */
	// font_desc: PangoFontDescription
	static native void vte_terminal_set_font(int handle, int font_desc);
	static native void vte_terminal_set_font_from_string(int handle, byte[] name);
	
	//return: PangoFontDescription.
	static native int vte_terminal_get_font(int handle);
	static native boolean vte_terminal_get_using_xft(int handle);
	static native void vte_terminal_set_allow_bold(int handle, boolean allow_bold);
	static native boolean vte_terminal_get_allow_bold(int handle);

	/* Check if the terminal is the current selection owner. */
	static native boolean vte_terminal_get_has_selection(int handle);

	/* Set the list of word chars, optionally using hyphens to specify ranges
 	* (to get a hyphen, place it first), and check if a character is in the
 	* range. */
	static native void vte_terminal_set_word_chars(int handle, byte[] spec);
	static native boolean vte_terminal_is_word_char(int handle, char c);

	/* Set what happens when the user strikes backspace or delete. */
	// binding VteTerminalEraseBinding
	static native void vte_terminal_set_backspace_binding(int handle, int binding);
	// binding VteTerminalEraseBinding
	static native void vte_terminal_set_delete_binding(int handle, int binding);

	/* Manipulate the autohide setting. */
	static native void vte_terminal_set_mouse_autohide(int handle, boolean setting);
	static native boolean vte_terminal_get_mouse_autohide(int handle);

	/* Reset the terminal, optionally clearing the tab stops and line history. */
	static native void vte_terminal_reset(int handle, boolean full, boolean clear_history);

	/* Read the contents of the terminal, using a callback function to determine
 	* if a particular location on the screen (0-based) is interesting enough to
 	* include.  Each byte in the returned string will have a corresponding
 	* struct vte_char_attributes in the passed GArray, if the array was not NULL.
 	* Note that it will have one entry per byte, not per character, so indexes
 	* should match up exactly. */
	/*static byte[] vte_terminal_get_text(int handle,
			    boolean(*is_selected)(int handle,
						   int column,
						   int row,
						   gpointer data),
			    gpointer data,
			    GArray *attributes);
	static byte[] vte_terminal_get_text_range(int handle,
				  int start_row, int start_col,
				  int end_row, int end_col,
				  boolean(*is_selected)(int handle,
							 int column,
							 int row,
							 gpointer data),
				  gpointer data,
				  GArray *attributes);*/
	static native void vte_terminal_get_cursor_position(int handle, int[] column, int[] row);

	/* Display string matching:  clear all matching expressions. */
	static native void vte_terminal_match_clear_all(int handle);

	/* Add a matching expression, returning the tag the widget assigns to that
	 * expression. */
	static native int vte_terminal_match_add(int handle, byte[] match);
	/* Remove a matching expression by tag. */
	static native void vte_terminal_match_remove(int handle, int tag);

	/* Check if a given cell on the screen contains part of a matched string.  If
 	* it does, return the string, and store the match tag in the optional tag
 	* argument. */
	static native byte[] vte_terminal_match_check(int handle, int column, int row, int[] tag);

	/* Set the emulation type.  Most of the time you won't need this. */
	static native void vte_terminal_set_emulation(int handle, byte[] emulation);
	static native byte[] vte_terminal_get_emulation(int handle);

	/* Set the character encoding.  Most of the time you won't need this. */
	static native void vte_terminal_set_encoding(int handle, byte[] codeset);
	static native byte[] vte_terminal_get_encoding(int handle);

	/* Get the contents of the status line. */
	static native byte [] vte_terminal_get_status_line(int handle);

	/* Get the padding the widget is using. */
	static native void vte_terminal_get_padding(int handle, int[] xpad, int[] ypad);

	/* Accessors for bindings. */
	//return: GtkAdjustment
	static native int vte_terminal_get_adjustment(int handle);
	static native int vte_terminal_get_char_width(int handle);
	static native int vte_terminal_get_char_height(int handle);
	static native int vte_terminal_get_char_descent(int handle);
	static native int vte_terminal_get_char_ascent(int handle);
	static native int vte_terminal_get_row_count(int handle);
	static native int vte_terminal_get_column_count(int handle);
	static native byte[] vte_terminal_get_window_title(int handle);
	static native byte[] vte_terminal_get_icon_title(int handle);
}
