/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtkhtml;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Vector;

import org.gnu.glib.EventMap;
import org.gnu.glib.GObject;
import org.gnu.gtk.event.GtkEventType;
import org.gnu.gtkhtml.event.HTMLDocumentEvent;
import org.gnu.gtkhtml.event.HTMLDocumentListener;

public class HTMLDocument extends GObject {
	/**
	 * Listeners for handling document events
	 */
	private Vector documentListeners = null;

	/**
	 * Map to collect the events.
	 */
	private static EventMap evtMap = new EventMap();

	static {
		addEvents(evtMap);
	}

	public HTMLDocument() {
		super(html_document_new());
		initializeEventHandlers();
	}
	
	public boolean openStream(String mimeType) {
		return html_document_open_stream(getHandle(), mimeType);
	}
	
	public void writeStream(String buffer) {
		htm_document_write_stream(getHandle(), buffer, buffer.length());
	}
	
	public void closeStream() {
		html_document_close_stream(getHandle());
	}
	
	public void clear() {
		html_document_clear(getHandle());
	}

	/**
	 * Load a file that is local to the process
	 * @param filename The name of the file to load
	 * @throws FileNotFoundException if the file doesn't exist
	 * @throws IOException if the file cannot be read
	 */
	public void loadFile(String filename) throws FileNotFoundException, IOException {
		File aFile = new File(filename);
		clear();
		openStream("text/html");
		try {
			BufferedReader in = new BufferedReader(new FileReader(aFile));
			String line;
			while ((line = in.readLine()) != null)
				if (line.length() > 0)
					writeStream(line);
		} finally {
			closeStream();
		}
	}
	
	/**
	 * Load a url that is on a remote web server.
	 * @param page The url of the remote page to display
	 * @throws MalformedURLException
	 * @throws IOException
	 */
	public void loadURL(String page) throws MalformedURLException, IOException {
		URL url = new URL(page);
		clear();
		openStream("text/html");
		try {
			BufferedInputStream in = new BufferedInputStream(url.openStream());
			byte[] buffer = new byte[4096];
			int bytesRead;
			while ((bytesRead = in.read(buffer)) != -1) {
				String data = new String(buffer);
				writeStream(data.trim());
			}
		} finally {
			closeStream();
		}
	}

	/**
	 * Register an object to handle document events.
	 * @see org.gnu.gtkhtml.event.HTMLDocumentListener
	 */
	public void addListener(HTMLDocumentListener listener) {
		// Don't add the listener a second time if it is in the Vector.
		int i = findListener(documentListeners, listener);
		if (i == -1) {
			if (null == documentListeners) {
				documentListeners = new Vector();
			}
			documentListeners.addElement(listener);
		}
	}

	/**
	 * Removes a listener
	 * @see #addListener(HTMLDocumentListener)
	 */
	public void removeListener(HTMLDocumentListener listener) {
		int i = findListener(documentListeners, listener);
		if (i > -1) {
			documentListeners.remove(i);
		}
		if (0 == documentListeners.size()) {
			documentListeners = null;
		}
	}

	/**
	 * Give us a way to locate a specific listener in a Vector.
	* @param list The Vector of listeners to search.
	* @param listener The object that is to be located in the Vector.
	* @return Returns the index of the listener in the Vector, or -1 if
	*                 the listener is not contained in the Vector.
	 */
	protected static int findListener(Vector list, Object listener) {
		if (null == list || null == listener)
			return -1;
		return list.indexOf(listener);
	}

	protected void fireDocumentEvent(HTMLDocumentEvent event) {
		if (null == documentListeners) {
			return;
		}
		int size = documentListeners.size();
		int i = 0;
		while (i < size) {
			HTMLDocumentListener dl = (HTMLDocumentListener)documentListeners.elementAt(i);
			dl.documentEvent(event);
			i++;
		}
	}

	protected void initializeEventHandlers() {
		evtMap.initialize(this);
	}

	public Class getEventListenerClass(String signal) {
		Class cls = evtMap.getEventListenerClass(signal);
		if (cls == null) cls = super.getEventListenerClass(signal);
		return cls;
	}

	public GtkEventType getEventType(String signal) {
		GtkEventType et = evtMap.getEventType(signal);
		if (et == null) et = super.getEventType(signal);
		return et;
	}

	/**
	* Implementation method to build an EventMap for this widget class.
	* Not useful (or supported) for application use.
	*/
	private static void addEvents(EventMap anEvtMap) {
		anEvtMap.addEvent("link_clicked", "handleLinkClicked", HTMLDocumentEvent.Type.LINK_CLICKED, HTMLDocumentListener.class);
		anEvtMap.addEvent("submit", "handleSubmit", HTMLDocumentEvent.Type.SUBMIT, HTMLDocumentListener.class);
	}
	
	protected void handleLinkClicked(String url) {
		HTMLDocumentEvent anEvent = new HTMLDocumentEvent(this, HTMLDocumentEvent.Type.LINK_CLICKED);
		anEvent.setURL(url);
		fireDocumentEvent(anEvent);
	}
	
	protected void handleSubmit(String method, String url, String encoding) {
		HTMLDocumentEvent anEvent = new HTMLDocumentEvent(this, HTMLDocumentEvent.Type.SUBMIT);
		anEvent.setMethod(method);
		anEvent.setURL(url);
		anEvent.setEncoding(encoding);
		fireDocumentEvent(anEvent);
	}

	/****************************************
	 * BEGINNING OF JNI CODE
	 ****************************************/
	native static final protected int html_document_get_type();
	native static final protected int html_document_new();
	native static final protected boolean html_document_open_stream(int document, String mimeType);
	native static final protected void htm_document_write_stream(int document, String buffer, int len);
	native static final protected void html_document_close_stream(int document);
	native static final protected void html_document_clear(int document);
	native static final protected void html_document_update_hover_node(int document, int node);
	native static final protected void html_document_update_active_node(int document, int node);
	native static final protected void html_document_update_focus_element(int document, int element);
	native static final protected void html_document_find_anchor(int document, String anchor);
	/****************************************
	 * END OF JNI CODE
	 ****************************************/
	static {
		System.loadLibrary("gtkhtmljava2.6");
	}
}
