
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <linux/kd.h>

#include "typedefs.h"
#include "leds.h"
#include "log.h"

/* console device */
static const char console[] = "/dev/console";

/* LED file descriptor */
static int fd = 0;

/* initial status of LEDs */
static char ledval;

/* error message */
static char errmsg[1024];

const char *led_open ()
{
   if (fd) return (NULL);

   if ((fd = open (console,O_RDONLY)) < 0)
	 {
		sprintf (errmsg,"couldn't open %s for reading",console);
		return (errmsg);
	 }

   if (ioctl (fd,KDGETLED,&ledval) < 0)
	 {
		sprintf (errmsg,"ioctl KDGETLED failed: %s",strerror (errno));
		close (fd);
		fd = 0;
		return (errmsg);
	 }

   return (NULL);
}

void led_close ()
{
   if (fd)
	 {
		led_clear (LED_NUM | LED_CAP | LED_SCR);
		close (fd);
		fd = 0;
	 }
}

static const char *set_led_status (char mask)
{
   if (!fd) return (NULL);

   if (ioctl (fd,KDSETLED,mask) < 0)
	 {
		sprintf (errmsg,"ioctl KDSETLED failed: %s",strerror (errno));
		return (errmsg);
	 }

   ledval = mask;

   return (NULL);
}

const char *led_set (int mask)
{
   if (mask & ~(LED_NUM | LED_CAP | LED_SCR))
	 {
		sprintf (errmsg,"%s: invalid argument",__FUNCTION__);
		return (errmsg);
	 }

   return (set_led_status (ledval | mask));
}

const char *led_clear (int mask)
{
   if (mask & ~(LED_NUM | LED_CAP | LED_SCR))
	 {
		sprintf (errmsg,"%s: invalid argument",__FUNCTION__);
		return (errmsg);
	 }

   return (set_led_status (ledval & ~mask));
}

