/****************************************************************************
 *                            ShmDFIOReader.cc
 *
 * Author: Matthew Ballance
 * Desc: 
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "ShmDFIOReader.h"
#include "ShmDFIOTrace.h"
#include <unistd.h>

#undef  DEBUG_SHM_DFIO_READER

#define FP stderr

#ifdef DEBUG_SHM_DFIO_READER
#define DBG_MSG(x) fprintf x ; fflush(FP)
#else
#define DBG_MSG(x)
#endif

/********************************************************************
 * ShmDFIOReader()
 ********************************************************************/
ShmDFIOReader::ShmDFIOReader(
        Tcl_Interp        *interp,
        int                argc,
        char             **argv) : ShmDFIO(interp, argc, argv)
{
    ok = false;

#ifndef __MINGW32__
    d_mappedSize = getpagesize();
#else
    d_mappedSize = 4096;
#endif

    d_file = new ShmDFIOFile(d_filename, true);

    if (!d_file->ok) {
        fprintf(stderr, "ERROR: Couldn't open file \"%s\"\n", d_filename);
        return;
    }

    d_file->map(d_mappedSize);
    d_header = (ShmDFIOHeader *)d_file->getMapBase();
    d_traceCount = 0;

    UpdateMapping();

    ok = true;
}

/********************************************************************
 * AddTraces()
 ********************************************************************/
void ShmDFIOReader::AddTraces()
{
    ShmDFIOPtr        traceDescPtr;
    ShmDFIOTraceDesc *traceDesc;
    ShmDFIOTrace     *trace;
    Uint32            i;
    Uint32            numTotalTraces = d_header->d_traceCount;

    /**** First, scan through the existing traces... ****/
    traceDescPtr = d_header->d_traceList;
    for (i=0; i<d_traceCount; i++) {
        traceDesc = (ShmDFIOTraceDesc *)ShmPtr2Ptr(traceDescPtr);

        if (!traceDesc->nextTrace) {
            fprintf(stderr, "ERROR: Premature end to trace list\n");
            fprintf(stderr, "\tExpecting %d - end at %d\n",
                    d_traceCount, i);
            return;
        }

        traceDescPtr = traceDesc->nextTrace;
    }

    /**** Now, iterate through each new trace... ****/
    for (i=0; i<(numTotalTraces-d_traceCount); i++) {
        if (!traceDescPtr) {
            fprintf(stderr, "ERROR: Premature end to new-trace list\n");
            return;
        }

        traceDesc = (ShmDFIOTraceDesc *)ShmPtr2Ptr(traceDescPtr);

        trace = new ShmDFIOTrace(this, traceDesc);
        addTrace(trace);

        DBG_MSG((FP, "Adding trace \"%s\"\n", traceDesc->name));

        traceDescPtr = traceDesc->nextTrace;
    }
    d_traceCount = numTotalTraces;
}

/********************************************************************
 * findTrace()
 ********************************************************************/
DFIOTrace *ShmDFIOReader::findTrace(Char *traceName)
{
    UpdateMapping();

    return base_findTrace(traceName);
}

/********************************************************************
 * UpdateMapping()
 ********************************************************************/
void ShmDFIOReader::UpdateMapping()
{
    Uint32     update = 0;
    DBG_MSG((FP, "----> UpdateMapping()\n"));

//    d_file->dump("reader", sizeof(ShmDFIOHeader)); 

    if (d_mappedSize != d_header->d_mappedSize) {
        DBG_MSG((FP, "\tMapped sizes don't match: "
            "d_mappedSize=%d, d_header->d_mappedSize=%d\n",
            d_mappedSize, d_header->d_mappedSize));
        /**** Remap to size specified in header ****/
        d_file->map(d_header->d_mappedSize);
        d_header = (ShmDFIOHeader *)d_file->getMapBase();
        d_mappedSize = d_header->d_mappedSize;
    }

    /**** If new traces were added, scan those in... ****/
    if (d_traceCount != d_header->d_traceCount) {
        update |= 1;
        AddTraces();
    }

    DBG_MSG((FP, "<---- UpdateMapping()\n"));

#if 0
    for (Uint32 i=0; i<traces->length(); i++) {
        if (((ShmDFIOTrace *)traces->idx(i))->Update()) {
            update |= 2;
        }
    }
#endif
}


