/****************************************************************************
 *                             ListWidget.cc
 *
 *
 * Author: Matthew Ballance
 * Desc:   Implements the list-widget core
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 ****************************************************************************/
#include "ListWidget.h"
#include "SdbReader.h"
#include "SdbSignal.h"
#include "DFIOTrace.h"

WIDGET_DECLS(ListWidget);

static void ListWidget_Display(ClientData clientData)
{
    ListWidget *listWidget = (ListWidget *)clientData;
    listWidget->DisplayWidget();
}

/**************************************************************
 * ListWidget()
 **************************************************************/
ListWidget::ListWidget(
        Tk_Window         tkwin,
        Tcl_Interp       *interp,
        int               argc,
        char            **argv)
{
    Uint32           i;

    instName       = new Char[strlen(argv[0])+1];
    strcpy(instName, argv[0]);

    okay = 0;
    this->tkwin    = tkwin;
    this->display  = Tk_Display(tkwin);
    this->interp   = interp;

    sdbr           = 0;
    sdbrName       = 0;

    for (i=0; i<TC_NumColors; i++) {
        GCs[i]     = 0;
        colors[i]  = 0;
    }

    background     = 0;
    tkfont         = 0;
    backingPixmap  = 0;
    displayPixmap  = 0;

    Tk_CreateEventHandler(tkwin, 
        ExposureMask|StructureNotifyMask|FocusChangeMask|
        ButtonPressMask|ButtonReleaseMask|Button1MotionMask,
        ListWidget_EventProc, (ClientData)this);

    if (Configure(argc-2, argv+2, 0) != TCL_OK) {
        fprintf(stderr, "ERROR: Configure failed\n");
        return;
    }

    Tk_SetWindowBackground(tkwin, Tk_3DBorderColor(background)->pixel);

    Tk_GeometryRequest(tkwin, 100, 100);
    Tk_SetInternalBorder(tkwin, 10);

    okay = 1;
}

/**************************************************************
 * DisplayWidget()
 **************************************************************/
void ListWidget::DisplayWidget(void)
{
    if (!tkwin || !Tk_IsMapped(tkwin)) {
        return;
    }

    UpdatePixmaps();

    Tk_Fill3DRectangle(tkwin, displayPixmap, background, 0, 0,
            widgetWidth, widgetHeight, 0, TK_RELIEF_FLAT);

    RepaintPixmap(backingPixmap);

    XCopyArea(display, backingPixmap, displayPixmap, 
            GCs[TC_TextColor], 0, 0,
            widgetWidth, widgetHeight, 0, 0);

    /**** Redraw goes here ****/

    XCopyArea(display, displayPixmap, Tk_WindowId(tkwin),
        GCs[TC_TextColor], 0, 0, widgetWidth, widgetHeight, 0, 0);

    CLR_FLAG(REDRAW_PENDING|SCROLL_X|SCROLL_Y|FORCE_REDRAW|APP_REQ_REDRAW);
    CLR_FLAG(CURSOR_CHANGE);
    CLR_FLAG(EXPOSE);
}

/**************************************************************
 * UpdatePixmaps()
 **************************************************************/
void ListWidget::UpdatePixmaps(void)
{
    Uint32        reallocYThresh = (3*backingPixmapHeight)/4;
    Uint32        reallocXThresh = (3*backingPixmapWidth)/4;


    if (!tkwin || !Tk_IsMapped(tkwin)) {
        return;
    }

    if (!displayPixmap || (Tk_Height(tkwin) != widgetHeight) ||
            (Tk_Width(tkwin) != widgetWidth) ) {

        widgetWidth   = Tk_Width(tkwin);
        widgetHeight  = Tk_Height(tkwin);

        if (displayPixmap) {
            Tk_FreePixmap(display, displayPixmap);
        }

        displayPixmap = Tk_GetPixmap(display, Tk_WindowId(tkwin),
                widgetWidth, widgetHeight, Tk_Depth(tkwin));
    }

    if ( !backingPixmap || 
            (Tk_Height(tkwin) > reallocYThresh) ||
            (Tk_Width(tkwin) > reallocXThresh)) {
        backingPixmapHeight = 2*widgetHeight;
        backingPixmapWidth  = 2*widgetWidth;

        if (backingPixmap) {
            Tk_FreePixmap(display, backingPixmap);
        }

        backingPixmap = Tk_GetPixmap(display, Tk_WindowId(tkwin),
                backingPixmapWidth, backingPixmapHeight, Tk_Depth(tkwin));
    }
}

/**************************************************************
 * RepaintPixmap()
 **************************************************************/
void ListWidget::RepaintPixmap(Pixmap pixmap)
{
    SdbSignal             *sig;
    Vector<DFIOValChg>    *vals;
    DFIOValChg            *val;
    Uint32                 i, ii, rows, col_max, line_height, max_rows, x;
    Uint32                 min_time;
    Tk_TextLayout          layout;
    Int32                  text_height, text_width;
    Char                   buf[1024];

    DFIOValChg           **valVect;
    Vector<DFIOValChg>   **valsVect;
    Uint32                *idxVect, len;
    String                 sv;


    if (!tkwin || !Tk_IsMapped(tkwin) || !sdbr) {
        return;
    }

    len = sdbr->d_signals.length();
    valsVect = new Vector<DFIOValChg> *[len];
    valVect = new DFIOValChg *[len];
    idxVect = new Uint32[len];
    memset(idxVect, 0, sizeof(Uint32)*len);

    Tk_Fill3DRectangle(tkwin, pixmap, background, 0, 0,
            widgetWidth, widgetHeight, 0, TK_RELIEF_FLAT);

    /**** First, figure out how tall a line is...
     ****/
    layout = Tk_ComputeTextLayout(tkfont, "Test String", -1, 0,
            TK_JUSTIFY_CENTER, 0, &text_width, &text_height);
    line_height = (text_height * 3) / 2;
    Tk_FreeTextLayout(layout);

    max_rows = widgetHeight / line_height;
    x = 10;

    /**** Load up a vector of values vectors...
     ****/
    for (i=0; i<len; i++) {
        sig = sdbr->d_signals.idx(i);
        valsVect[i] = sig->dfioTrace->getValue(0, 0xFFFFFFFF, 0);
    }

#ifdef UNDEFINED 
    ii = 0;
    while (ii < max_rows) {
        min_time = 0xFFFFFFFF;

        /**** First, find the min time... ****/
        for (i=0; i<len; i++) {
            if ((val = valsVect[i]->idx(idxVect[i]))) {
                if (val->changeTime > min_time) {
                    min_time = val->changeTime;
                }
            }
        }

        /**** Display the time for this row ****/
        layout = Tk_ComputeTextLayout(tkfont, buf, -1, 0,
                TK_JUSTIFY_CENTER, 0, &text_width, &text_height);

        /**** Now, display any signals with changeTime == min_time
         **** Increment the val-index of any signal whose value we
         **** display
         ****/
        for (i=0; i<len; i++) {
            if ((val = valsVect[i]->idx(idxVect[i]))) {
                if (val->changeTime == min_time) {
                    /* Display */
                    layout = Tk_ComputeTextLayout(tkfont, buf, -1, 0,
                        TK_JUSTIFY_CENTER,0, &text_width, &text_height);

                    /* Bump idx */
                    idxVect[i]++;
                }
            }
        }
    }
#endif /* UNDEFINED */

    for (i=0; i<sdbr->d_signals.length(); i++) {
        sig = sdbr->d_signals.idx(i);

        col_max = 0;
        vals = sig->dfioTrace->getValue(0, 0xFFFFFFFF, DFIOTrace::GVF_Default);
        for (ii=0; ((ii<vals->length()) && (ii<max_rows)); ii++) {
            val = vals->idx(ii);
            sig->FormatValue(val, sv);

            layout = Tk_ComputeTextLayout(tkfont, sv.value(), 
                    sv.length(), 0,
                    TK_JUSTIFY_CENTER,0, &text_width, &text_height);
            if (text_width > col_max) {
                col_max = text_width;
            }

            Tk_DrawTextLayout(display, pixmap, GCs[TC_TextColor],
                    layout, x, line_height*ii, 0, -1);
            Tk_FreeTextLayout(layout);
        }

        x += col_max + 10;
    }

    delete valVect;
    delete valsVect;
    delete idxVect;
}

/**************************************************************
 * EventProc()
 **************************************************************/
void ListWidget::EventProc(XEvent    *eventPtr)
{
    switch(eventPtr->type) {
        case Expose:
            if (backingPixmapValid && (widgetWidth == Tk_Width(tkwin)) &&
                    (widgetHeight == Tk_Height(tkwin))) {
                    XCopyArea(display, displayPixmap, Tk_WindowId(tkwin),
                        GCs[TC_TextColor],
                        eventPtr->xexpose.x,     eventPtr->xexpose.y,
                        eventPtr->xexpose.width, eventPtr->xexpose.height,
                        eventPtr->xexpose.x,     eventPtr->xexpose.y);
            } else if (eventPtr->xexpose.count == 0) {
                SetupRedraw(EXPOSE);
            }
            break;

        case DestroyNotify:
            Tcl_DeleteCommand(interp, instName);
            Tk_DeleteEventHandler(tkwin, 
                ExposureMask|StructureNotifyMask|FocusChangeMask|
                ButtonPressMask|ButtonReleaseMask|Button1MotionMask,
                ListWidget_EventProc, (ClientData)this);

            tkwin = NULL;

            if (IS_FLAG_SET(REDRAW_PENDING)) {
                Tk_CancelIdleCall(ListWidget_Display, this);
            }
            CLR_FLAG(DRAW_FLAGS);
            Tk_EventuallyFree((ClientData)this, ListWidget_Destroy);
            break;
    }
}

/**************************************************************
 * InstCmd()
 **************************************************************/
int ListWidget::InstCmd(
        int          argc,
        char       **argv)
{
    Int32          cmd;
/*    char         **argv = (char **)argv_p; */

    cmd = CmdSwitch(lw_cmds, argv[1]);

    switch (cmd) {
        case LWC_Config:
            return Configure(argc-2, &argv[2], TK_CONFIG_ARGV_ONLY);
            break;

        case LWC_Cget:
            return Tk_ConfigureValue(interp, tkwin, getConfigSpec(),
                    (char *)this, argv[2], 0);
            break;

        case LWC_Redraw:
            SetupRedraw(FORCE_REDRAW);
            break;

        default:
            Tcl_AppendResult(interp, "invalid ListWidget subcommand", 0);
            return TCL_ERROR;
            break;
    }
    return TCL_OK;
}

/******************************************************************
 * SetupRedraw()
 ******************************************************************/
void ListWidget::SetupRedraw(Uint32  flags)
{
    redrawFlags |= flags;

    if (IS_FLAG_SET(FORCE_REDRAW|APP_REQ_REDRAW)) {
        backingPixmapValid = 0;
    }

    if (tkwin && Tk_IsMapped(tkwin) && !(flags & REDRAW_PENDING)) {
        Tk_DoWhenIdle(ListWidget_Display, (ClientData)this);
        redrawFlags |= REDRAW_PENDING;
    }
}


/******************************************************************
 * Configure()
 ******************************************************************/
int ListWidget::Configure(int argc, char **argv, int flags)
{
    if (Tk_ConfigureWidget(interp, tkwin, getConfigSpec(), argc, 
                (const char **)argv, (char *)this, flags) != TCL_OK) {
        fprintf(stderr, "ERROR: ConfigureWidget error\n");
        return TCL_ERROR;
    }

    /**** Lookup the specified SDBR
     ****/
    if (optionSpecified(Opt_Sdbr) && sdbrName) {
        sdbr = (SdbReader *)WidgetMgr_GetObjHandle(interp, "SdbReader", 
                sdbrName);

        if (!sdbr) {
            Tcl_AppendResult(interp, "SdbReader ", sdbrName, 
                    " doesn't exist", 0);
            return TCL_ERROR;
        }
    }

    if (optionSpecified(Opt_Foreground) || !flags) {
        Uint32 i;
        XGCValues    gcValues;

        for (i=0; i<TC_NumColors; i++) {
            gcValues.background = Tk_3DBorderColor(background)->pixel;
            gcValues.foreground = colors[i]->pixel;
            gcValues.font       = Tk_FontId(tkfont);
            gcValues.graphics_exposures = False;

            if (GCs[i]) {
                Tk_FreeGC(display, GCs[i]);
            }

            GCs[i] = Tk_GetGC(tkwin, 
                    GCBackground|GCForeground|GCFont|GCGraphicsExposures,
                    &gcValues);
        }
    }

    SetupRedraw(FORCE_REDRAW);

    return TCL_OK;
}

WIDGET_FUNCS(ListWidget, "list_widget", 1.0);

extern "C" int ListTitle_Init(Tcl_Interp *interp);

extern "C" int List_widget_Init(Tcl_Interp *interp) {
    ListWidget_Init(interp);
    ListTitle_Init(interp);
    return TCL_OK;
}

extern "C" int List_widget_SafeInit(Tcl_Interp *interp) {
	return List_widget_Init(interp);
}

