#****************************************************************************
#*                             sdb_cmds.tcl
#*
#* Author: Matthew Ballance
#* Desc:   Implements user-level commands for managing Signal DataBases
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#****************************************************************************

#********************************************************************
#* prv_sdb_findtype
#********************************************************************
proc prv_sdb_findtype {type} {

    foreach filter [dfio_mgr filter_list] {
        if {[$filter id] == $type} {
            return $filter
        }
    }
    return ""
}

#********************************************************************
#* prv_sdb_guesstype
#********************************************************************
proc prv_sdb_guesstype {file} {

    set fs [file extension $file]
    set filt ""

    foreach filter [dfio_mgr filter_list] {
        foreach s [$filter suffixes] {
            if {$s == $fs} {
                set filt $filter
                break
            }
        }

        if {$filt != ""} {
            break
        }
    }

    return $filt
}

#********************************************************************
#* sdb_open
#*
#* open <SdbId> <FileName> ?-type FileType?
#********************************************************************
proc sdb_open {args} {
    set args [lindex $args 0]
    if {[llength $args] < 2} {
        error "too few args: sdb open <SdbId> <FileName> ?-type FileType?"
    }

    set type ""

    set sdb_id   [lindex $args 0]
    set filename [lindex $args 1]

    set opts [lrange $args 2 end]
    while {[llength $opts] > 0} {
        set opt [lindex $opts 0]
        set shift 1

        if {$opt == "-type"} {
            set shift 2
            set type [lindex $opts 1]
        } 
        set opts [lrange $opts $shift end]
    }

    set filter ""

    if {$type != ""} {
        set filter [prv_sdb_findtype $type]
    } else {
        set filter [prv_sdb_guesstype $filename]
    }

    if {$filter == ""} {
        error "Filter cannot be found for file \"$filename\""
    }

    foreach sdb_i [sdb_mgr list] {
        if {[$sdb_i cget -sdb_id] == $sdb_id} {
            error "SDB named \"$sdb_id\" already exists"
        }
    }

    set dfio [dfio_mgr create ::DFIO.$sdb_id.dfio]
    set ddb  [ddb             ::DDB.$sdb_id.ddb  ]

    if {[catch {$filter import $dfio $filename} res]} {
        ivi_puts "ERROR: while importing \"$filename\": $res"
        $dfio delete 
        error "while importing \"$filename\": $res"
    } else {
        ivi_puts "Opened SDB \"$sdb_id\" from file \"$filename\""
    }

    $ddb configure -dfio $dfio

    set sdb  [sdb_mgr create ::SDB.$sdb_id -dfio $dfio -ddb $ddb \
        -sdb_id $sdb_id]

    return $sdb
}

#********************************************************************
#* sdb_pick_id
#********************************************************************
proc sdb_pick_id {} {
    #**** Select an SDB name...
    for {set i 1} {$i < 256} {incr i} {
        set sdb_name "SigData$i"

        set ok 1
        foreach sdb [sdb_mgr list] {
            if {[$sdb cget -sdb_id] == $sdb_name} {
                set ok 0
                break
            }
        }
        if {$ok == 1} {
            break
        }
    }

    return $sdb_name
}

#********************************************************************
#* sdb_save
#*
#* save <SdbId> <FileName> ?-type FileType?
#********************************************************************
proc sdb_save {args} {

    set args [lindex $args 0]
    if {[llength $args] < 2} {
        error "too few args: sdb save <SdbId> <FileName> ?-type FileType?"
    }

    set type ""
    set opts [lrange $args 2 end]
    while {[llength $opts] > 0} {
        set opt [lindex $opts 0]
        set shift 1

        if {$opt == "-type"} {
            set shift 2
            set type [lindex $opts 1]
        } 
        set opts [lrange $opts $shift end]
    }

    set sdb_id   [lindex $args 0]
    set filename [lindex $args 1]

    if {$type != ""} {
        set filter [prv_sdb_findtype $type]
    } else {
        set filter [prv_sdb_guesstype $filename]
    }

    if {$filter == ""} {
        error "Filter cannot be found for file \"$filename\""
    }

    #**** Find the sdb
    set sdb ""
    foreach sdb_i [sdb_mgr list] {
        if {[$sdb_i cget -sdb_id] == $sdb_id} {
            set sdb $sdb_i
            break
        }
    }

    if {$sdb == ""} {
        error "no SDB named \"$sdb_id\""
    }

    set dfio [$sdb cget -dfio]

    if {[catch {$filter export $dfio $filename} res]} {
        error "while saving \"$filename\": \"$res\""
    } else {
        ivi_puts "Saved SDB \"$sdb_id\" to file \"$filename\""
    }
}

#********************************************************************
#* sdb_is_valid
#*
#* sdb is_valid <FileName> ?-type FileType?
#********************************************************************
proc sdb_is_valid args {
    set args [lindex $args 0]

    if {[llength $args] < 2} {
    }
}

#********************************************************************
#* sdb
#********************************************************************
proc sdb args {
    set sub_cmd [lindex $args 0]

    switch $sub_cmd {
        open {
            sdb_open [lrange $args 1 end]
        }

        pick_id {
            sdb_pick_id 
        }

        save {
            sdb_save [lrange $args 1 end]
        }

        is_of_type {
            sdb_of_type [lrange $args 1 end]
        }

        close {

        }

        default {
            error "Unknown SDB sub-command $sub_cmd"
        }
    }
}

