#****************************************************************************
#*                               knob.tcl
#*
#* Author: Matthew Ballance
#* Desc:
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#****************************************************************************

namespace eval Knob {

    variable d_configspec {
        { "-knob_radius"           32       }
        { "-dot_radius"            5        }
        { "-pady"                  6        }
        { "-padx"                  6        }
        { "-knob_color"            "grey60" }
        { "-dot_color"             "black"  }
        { "-bg"                    "grey95" }
        { "-command"               ""       }
        { "-pos_dir"               "cw"     }
        { "-origin"                0        }
        { "-left_limit"            0        }
        { "-right_limit"           0        }
        { "-pos_limit"             -1       }
        { "-neg_limit"             -1       }
    }

    variable d_pi 3.14159265359
    variable d_2pi [expr $d_pi * 2]
    variable d_32pi [expr ($d_pi * 3) / 2.0]
}


#********************************************************************
#* Knob
#********************************************************************
proc Knob::Knob {path args} {
    variable d_configspec

    set path [canvas $path -borderwidth 0]

    set canvas ::$path:cmd

    rename $path ::$path:cmd
    proc ::$path {cmd args} "return \[eval Knob::cmd $path \$cmd \$args\]"

    array set $path {_dummy _dummy}
    upvar #0 $path data

    BaseClass::config_init $path $d_configspec
    eval BaseClass::configure $path $args

    set data(w:canvas) $canvas

    set data(w:circle) [$data(w:canvas) create oval \
        $data(-padx) \
        $data(-pady) \
        [expr $data(-padx) + (2 * $data(-knob_radius))] \
        [expr $data(-pady) + (2 * $data(-knob_radius))] \
        -fill $data(-knob_color) \
    ]

    set data(w:dot)   [$data(w:canvas) create oval \
        0 0 \
        [expr 2 * $data(-dot_radius)] \
        [expr 2 * $data(-dot_radius)] \
        -fill $data(-dot_color)       \
    ]
    
    set data(dot_x) 0
    set data(dot_y) 0

     $data(w:canvas) configure \
        -width [expr (2 * $data(-knob_radius)) + (2 * $data(-padx))] \
        -height [expr (2 * $data(-knob_radius)) + (2 * $data(-pady))]\
        -bg $data(-bg)

    SetPos $path $data(-origin)

    set data(curr_x) 0
    set data(curr_y) 0

    $data(w:canvas) bind $data(w:circle) <1> \
        "Knob::ButtonBind $path %x %y"
    $data(w:canvas) bind $data(w:dot) <1> \
        "Knob::ButtonBind $path %x %y"

    $data(w:canvas) bind $data(w:circle) <B1-Motion> \
        "Knob::ButtonMotionBind $path %x %y"
    $data(w:canvas) bind $data(w:dot) <B1-Motion> \
        "Knob::ButtonMotionBind $path %x %y"

    return $path
}

#********************************************************************
#* cmd
#********************************************************************
proc Knob::cmd {path cmd args} {

    switch $cmd {
        config -
        configure {
            eval Configure $args
        }

        cget {
            eval BaseClass::cget $args
        }

        set_pos {
            eval Knob::SetPos $path [lindex $args 0]
        }

        default {
            error "Unknown knob sub-command $cmd"
        }
    }
}

#********************************************************************
#* Configure
#********************************************************************
proc Knob::Configure {path args} {
    eval BaseClass::configure $path $args
}

#********************************************************************
#* SetPos
#********************************************************************
proc Knob::SetPos {w rad} {
    upvar #0 $w data

    set data(curr_rad) $rad

    set new_pos [FindDestXY $w $rad]

    set dx [expr [lindex $new_pos 0] - $data(dot_x)]
    set dy [expr [lindex $new_pos 1] - $data(dot_y)]

    set data(dot_x) [lindex $new_pos 0]
    set data(dot_y) [lindex $new_pos 1]

    $data(w:canvas) move $data(w:dot) $dx $dy
}

#********************************************************************
#* FindDestXY
#********************************************************************
proc Knob::FindDestXY {w rad} {
    upvar #0 $w data 

    set orbit [expr ($data(-knob_radius)*3)/4]

    if {[expr ($orbit + $data(-knob_radius))] >= $data(-knob_radius)} {
        incr orbit -1
    }

    set dx [expr cos($rad) * $orbit]
    set dy [expr sin($rad) * $orbit]

    set dx [expr $dx + $data(-padx) + $data(-knob_radius)]
    set dy [expr $data(-pady) + $data(-knob_radius) - $dy]

    set dx [expr $dx - $data(-dot_radius)]
    set dy [expr $dy - $data(-dot_radius)]

    return [list $dx $dy]
}

#********************************************************************
#* ButtonBind
#********************************************************************
proc Knob::ButtonBind {w x y} {
    upvar #0 $w data

    set data(curr_x) $x
    set data(curr_y) $y
}

#********************************************************************
#* ButtonMotionBind
#********************************************************************
proc Knob::ButtonMotionBind {w x y} {
    upvar #0 $w data
    variable d_pi
    variable d_2pi
    variable d_32pi

    set dx [expr $x - $data(curr_x)]
    set dy [expr $y - $data(curr_y)]

    set tx [expr cos($data(curr_rad)) * $data(-knob_radius)]
    set ty [expr sin($data(curr_rad)) * $data(-knob_radius)]

    set rx [expr $tx + $dx]
    set ry [expr $ty - $dy]

    set new_rad [expr atan2($ry , $rx)]

    #**** Fixup the fact that atan only handles two quadrants
    if {$new_rad < 0} {
        set new_rad [expr (2 * $d_pi) + $new_rad]
    } 

    if {$data(-left_limit) > 0} {
        set limit [expr $data(-left_limit) + $data(-origin)]
        puts "limit=$limit ; new_rad=$new_rad"
        if {$new_rad > $limit && $new_rad < $d_32pi} {
           puts "limited to left"
            set new_rad $limit
        }
    }

    if {$data(-right_limit) > 0} {
        set limit [expr $d_2pi - $data(-right_limit)]
        if {$new_rad < $limit} {
            set new_rad $limit
        }
    }

    set old_rad $data(curr_rad)
    SetPos $w $new_rad

    set data(curr_x) $x
    set data(curr_y) $y

    set delta 0

    #**** First, figure out the delta 
    if {$new_rad > $old_rad} {
        set delta [expr $new_rad - $old_rad]
        if {$delta > $d_pi} {
            #**** CW wrap... 
            set delta [expr ($d_2pi - $new_rad) + $old_rad]
            if {$data(-pos_dir) == "ccw"} {
                set delta [expr - $delta]
            }
        } else {
            if {$data(-pos_dir) == "cw"} {
                set delta [expr - $delta]
            }
        }
    } elseif {$new_rad < $old_rad} {
        set delta [expr $old_rad - $new_rad]
        if {$delta > $d_pi} {
            set delta [expr ($d_2pi - $old_rad) + $new_rad]
            if {$data(-pos_dir) == "cw"} {
                set delta [expr - $delta]
            }
        } else {
            if {$data(-pos_dir) == "ccw"} {
                set delta [expr - $delta]
            }
        }
    }

    set norm_rad [expr $new_rad - $data(-origin)]

    #**** Now that we have a delta, check the boundaries...
    if {$delta > 0} {
        if {($data(-pos_limit) >= 0) && ($norm_rad > $data(-pos_limit))} {
            puts "hit pos limit: delta=$delta"
#            set delta [expr $delta - ($data(-pos_limit)-$norm_rad)]
        }
    } elseif {$delta < 0} {

    }
    
    if {($data(-command) != "") && ($delta != 0)} {
        eval $data(-command) $norm_rad $delta
    }
}

#********************************************************************
#* rad2deg
#********************************************************************
proc Knob::rad2deg {rad} {
    variable d_pi
    return [expr (360.0 / (2 * $d_pi)) * $rad]
}

#********************************************************************
#* deg2rad
#********************************************************************
proc Knob::deg2rad {deg} {
    variable d_pi
    return [expr ((2 * $d_pi) / 360.0) * $deg]
}

