/**************************************************************************
 *                          WidgetCommon.h
 *
 * Author: Matthew Ballance
 * Desc:   Implements the basics of creating a Tk widget...
 *
 * Required member functions
 *
 * WidgetName::WidgetName(Tk_Window win, Tcl_Interp *interp,
 *     int argc, char **argv)
 *
 * WidgetName::EventProc(XEvent *eventPtr)
 *
 * WidgetName::InstCmd(int argc, char **argv)
 *
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 **************************************************************************/

#ifndef INCLUDED_WIDGET_COMMON_H
#define INCLUDED_WIDGET_COMMON_H

/**************************************************************************
 * Widget include files
 **************************************************************************/
#include <tcl.h>
#include <tk.h>
#include <stdlib.h>
#include <string.h>
#include "WidgetManager.h"

/**************************************************************************
 * WIDGET_DELCS()
 **************************************************************************/
#define WIDGET_DECLS(WidgetName) \
    static int TclCmd_ ## WidgetName ( ClientData clientData, \
            Tcl_Interp *interp, int argc, const char **argv); \
    static void WidgetName ## _EventProc( ClientData clientData, \
            XEvent *eventPtr); \
    static void WidgetName ## _Destroy(char *clientData); \
    extern "C" int WidgetName ## _Init (Tcl_Interp *interp);



/**************************************************************************
 * WIDGET_CONSTRUCTOR()
 *
 * Declears the TCL procedure that constructs the window.
 *
 * NOTE:
 *    Requires that the class contain a member named 'okay' that is set 
 * true when widget constructs okay.
 **************************************************************************/
#define WIDGET_CONSTRUCTOR(WidgetName) \
    static int WidgetName ## _InstCmd(ClientData, Tcl_Interp *, \
                int, const char **); \
    static int TclCmd_ ## WidgetName ( ClientData clientData, \
            Tcl_Interp    *interp, \
            int            argc, \
            const char   **argv_p) \
    { \
        Tk_Window    main = (Tk_Window)clientData; \
        WidgetName  *window; \
        Tk_Window    tkwin; \
        char       **argv = (char **)argv_p; \
        \
        \
        if (argc < 2) { \
            Tcl_AppendResult(interp, "wrong # args", NULL); \
            return TCL_ERROR; \
        } \
        \
        tkwin = Tk_CreateWindowFromPath(interp, Tk_MainWindow(interp), \
                argv[1], (char *)NULL); \
        if (!tkwin) { \
            fprintf(stderr, "ERROR: Tk_CreateWindowFromPath(%s) failed\n", \
                # WidgetName); \
            return TCL_ERROR; \
        } \
        window = new WidgetName(tkwin, interp, argc, argv); \
        \
        if (!window->okay) { \
            Tcl_AppendResult(interp, "ERROR: " # WidgetName  \
                    "construction failed!!", 0); \
            return TCL_ERROR; \
        } \
            Tcl_CreateCommand(interp, argv[1],  \
                    (Tcl_CmdProc *)WidgetName ##_InstCmd, \
                    (voidPtr)window, NULL); \
            Tk_SetClass(tkwin, # WidgetName); \
            \
        WidgetMgr_AddInst(interp, # WidgetName, argv[1], window); \
            \
        Tcl_SetResult(interp, argv[1], TCL_VOLATILE); \
        return TCL_OK; \
    } \

/**************************************************************************
 * WIDGET_EVENTPROC()
 **************************************************************************/
#define WIDGET_EVENTPROC(WidgetName) \
    static void WidgetName ## _EventProc( ClientData clientData, \
            XEvent *eventPtr) \
    { \
        WidgetName *widget = (WidgetName *)clientData; \
        widget->EventProc(eventPtr); \
    } 

/**************************************************************************
 * WIDGET_DESTROY()
 **************************************************************************/
#define WIDGET_DESTROY(WidgetName) \
    static void WidgetName ## _Destroy(char *clientData) \
    { \
        WidgetName *widget = (WidgetName *)clientData; \
        delete widget; \
    } \


/**************************************************************************
 * WIDGET_INSTCMD()
 **************************************************************************/
#define WIDGET_INSTCMD(WidgetName) \
    static int WidgetName ## _InstCmd(ClientData clientData, \
            Tcl_Interp *interp, int argc, const char *argv[]) \
    { \
        WidgetName *widget = (WidgetName *)clientData; \
        return widget->InstCmd(argc, (char **)argv); \
    } 

/**************************************************************************
 * WIDGET_INIT
 **************************************************************************/
#define WIDGET_INIT(WidgetName, WidgetCmd, Version) \
    extern "C" int WidgetName ## _Init(Tcl_Interp *interp) { \
        if (Tcl_InitStubs(interp, "8.1", 0) == NULL) { \
            return TCL_ERROR; \
        } \
        \
        WidgetMgr_Init(interp); \
        Tcl_CreateCommand(interp, WidgetCmd, \
                (Tcl_CmdProc *)TclCmd_ ## WidgetName, \
                NULL, NULL); \
        Tcl_PkgProvide(interp, # WidgetName, # Version); \
        WidgetMgr_AddType(interp, # WidgetName ); \
        return TCL_OK; \
    }


/**************************************************************************
 * WIDGET_FUNCS()
 *
 * Declares all required widget functions
 *
 * NOTE: WidgetName is an identifier
 *       WidgetCmd is a string
 *       Version is a floating-point number
 **************************************************************************/
#define WIDGET_FUNCS(WidgetName, WidgetCmd, Version) \
    WIDGET_CONSTRUCTOR(WidgetName); \
    WIDGET_EVENTPROC(WidgetName); \
    WIDGET_DESTROY(WidgetName); \
    WIDGET_INSTCMD(WidgetName); \
    WIDGET_INIT(WidgetName, WidgetCmd, Version); 


#endif /* INCLUDED_WIDGET_COMMON_H */


