/*****************************************************************
 *                          VExpr.cc
 * Author: Matthew Ballance
 * Desc:   Implements a verilog-expression parser using the
 *         BitVector class...
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************/
#include "VExpr.h"
#include <tcl.h>

/*****************************************************************
 *
 *****************************************************************/
VExpr::VExpr(void)
{
}

/*****************************************************************
 * opPrec
 *****************************************************************/
Int32 VExpr::opPrec(VExpr_Op op)
{
    static struct {
        VExpr_Op      op;
        Uint32        prec;
    } precTab[] = {
          { VO_Unknown,               0 },
          { VO_Plus,                  5 },
          { VO_Minus,                 5 },
          { VO_Times,                 4 },
          { VO_Div,                   4 },
          { VO_BitOr,                11 },
          { VO_BitAnd,                9 },
          { VO_BitXor,               10 },
          { VO_BitNot,                2 },
          { VO_LogOr,                13 },
          { VO_LogAnd,               12 },
          { VO_LogXor,               12 },
          { VO_LogNot,                2 },
          { VO_Rparen,                1 },
          { VO_Lparen,                1 },
          { VO_UnPlus,                2 },
          { VO_EqTest,                8 },
          { VO_NeqTest,               8 },
          { VO_LeftShift,             6 },
          { VO_RightShift,            6 },
          { VO_LessTest,              7 },
          { VO_LessEqTest,            7 },
          { VO_GreaterTest,           7 },
          { VO_GreaterEqTest,         7 },
          { VO_NumOps,                0 }
    };

    if (op >= VO_Plus && op < VO_NumOps) {
        return precTab[op].prec;
    } else {
        return -1;
    }
}

/*****************************************************************
 * strToOp
 *****************************************************************/
Uint32 VExpr::strToOp(Char *str)
{
    Uint32    idx = 0;
    static struct {
        Char        *str;
        VExpr_Op     op;
    } strOpTab[] = {
        { "+",             VO_Plus         },
        { "-",             VO_Minus        },
        { "*",             VO_Times        },
        { "/",             VO_Div          },
        { "|",             VO_BitOr        },
        { "&",             VO_BitAnd       },
        { "^",             VO_BitXor       },
        { "~",             VO_BitNot       },
        { "||",            VO_LogOr        },
        { "&&",            VO_LogAnd       },
        { "^^",            VO_LogXor       },
        { "!",             VO_LogNot       },
        { ")",             VO_Rparen       },
        { "(",             VO_Lparen       },
        { "==",            VO_EqTest       },
        { "!=",            VO_NeqTest      },
        { "<<",            VO_LeftShift        },
        { ">>",            VO_RightShift       },
        { "<",             VO_LessTest         },
        { "<=",            VO_LessEqTest       },
        { ">",             VO_GreaterTest      },
        { ">=",            VO_GreaterEqTest    },
        { "",              (VExpr_Op)0         }
    };


    while (strOpTab[idx].str[0]) {
        if (String::equal(strOpTab[idx].str, str)) {
            return strOpTab[idx].op;
        }
        idx++;
    }

    return VO_Unknown;
}


/*****************************************************************
 * pushOperand()
 *****************************************************************/
void VExpr::pushOperand(BitVector  *op)
{

}

/*****************************************************************
 * executeOp
 *****************************************************************/
BitVector *VExpr::executeOp(BitVector *op1, VExpr_Op op, BitVector *op2)
{
    BitVector *result;
    if (bitVectFreeStore.depth()) {
        result = bitVectFreeStore.pop();
    } else {
        result = new BitVector();
    }

    switch (op) {
        case VO_Plus: *result = *op1 + *op2; break;

        case VO_Minus: *result = *op1 - *op2; break;

        case VO_Times:
            break;

        case VO_Div:
            break;

        case VO_BitOr: *result = *op1 | *op2; break;

        case VO_BitAnd: *result = *op1 & *op2; break;

        case VO_BitXor: *result = *op1 ^ *op2; break;

        case VO_BitNot: *result = ~*op1; break;

        case VO_LogOr:
            break;

        case VO_LogAnd:
            break;

        case VO_LogXor:
            break;

        case VO_LogNot:
            break;

        case VO_Rparen:
            break;

        case VO_Lparen:
            break;

        case VO_UnPlus:
            break;

        case VO_EqTest:
            break;

        case VO_NeqTest:
            break;

        case VO_LeftShift:
            break;

        case VO_RightShift:
            break;

        case VO_LessTest:
            break;

        case VO_LessEqTest:
            break;
        case VO_GreaterTest:
            break;
        case VO_GreaterEqTest:
             break;
        default:
            break;
    }

}

/*****************************************************************
 * pushOperator()
 *****************************************************************/
void VExpr::pushOperator(VExpr_Op op)
{
    VExpr_Op  *top; 

    /**** If we have at least two operands on the stack, then we can
     **** try to reduce...
     ****/
    if (operandStack.depth() >= 2) {
        /**** Now, test precedence. This is at least the second operator
         ****/
        top = operatorStack.top();

        /**** Stack-top being less-equal than op means that the new 
         **** operator is either same or lower priority than stack-top. 
         **** This means that we can reduce the operation...
         ****/
        if (opPrec(op) >= opPrec(*top)) {


            return;
        } 
    }
    
    /**** Okay, go ahead and push the operator... ****/
    if (operatorFreeStore.depth()) {
        top = operatorFreeStore.pop();
    } else {
        top = new VExpr_Op;
    }
    *top = op;
    operatorStack.push(top);
}


/*****************************************************************
 * VExpr_TclCmd()
 *****************************************************************/
static int VExpr_TclCmd(
        ClientData         clientData,
        Tcl_Interp        *interp,
        int                objc,
        Tcl_Obj           *const objv[])
{
    Uint32 i;

    fprintf(stderr, "----> VExpr_TclCmd()\n");
    for (i=0; i<objc; i++) {
        fprintf(stderr, "\t%s\n", Tcl_GetString(objv[i]));
    }

    return TCL_OK;
}


extern "C" int VExpr_Init(Tcl_Interp *interp)
{
    Tcl_CreateObjCommand(interp, "vexpr", VExpr_TclCmd, 0, 0);
    return TCL_OK;
}


