/*****************************************************************************
 *                               SdbReader.h
 * Author: Matthew Ballance
 * Desc:   Implements a signal reader...
 *
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *****************************************************************************/
#ifndef INCLUDED_SDB_READER_H
#define INCLUDED_SDB_READER_H

#include <tcl.h>
#include <tk.h>
#include <string.h>
#include "types.h"
#include "vector.h"
#include "ivi_String.h"
#include "CmdSwitcher.h"
#include "HashTable.h"

typedef enum {
    SRC_Config = 1,
    SRC_Cget,
    SRC_Add,
    SRC_Del,
    SRC_List,
    SRC_AddCursor,
    SRC_GetCursor,
    SRC_GetCursorList,
    SRC_AddSep,
    SRC_Move,
    SRC_ClearCutBuffer,
    SRC_AppendCutBuffer,
    SRC_PutCutBuffer,
    SRC_LastCmd
} SdbReaderCmds;

static CmdSwitchStruct sdbReaderCmds[] = {
    {"config",          SRC_Config        },     /* D */
    {"configure",       SRC_Config        },     /* D */
    {"cget",            SRC_Cget          },     /* D */
    {"add",             SRC_Add           },     /* D */
    {"del",             SRC_Del           },     /* D */
    {"delete",          SRC_Del           },     /* D */
    {"list",            SRC_List          },     /* D */
    {"add_cursor",      SRC_AddCursor     },     /* D */
    {"get_cursor",      SRC_GetCursor     },     /* D */
    {"get_cursor_list", SRC_GetCursorList },     /* D */
    {"add_sep",         SRC_AddSep        },     /* D */
    {"add_separator",   SRC_AddSep        },     /* D */
    {"move",            SRC_Move          },     /* D */
    {"clear_cut_buffer", SRC_ClearCutBuffer},   
    {"append_cut_buffer", SRC_AppendCutBuffer},
    {"put_cut_buffer",    SRC_PutCutBuffer   },
    {"", 0}
    };

class CursorMgr;
class SdbSignal;
class SigDB;
class SdbDesc;

class SdbReader {

public:
    SdbReader(Tcl_Interp *interp, Uint32 argc, Char **argv);

   ~SdbReader(void);

    Int32 InstCmd(Uint32 argc, Char **argv);

    inline Char *getInstName(void) const {return instName.value();}



    Vector<SdbSignal>     d_signals;
    Vector<SdbSignal>     d_tmplist;
    Vector<SdbSignal>     d_cutBuffer;

    void  getSigList(void);

    /****************************************************************
     * addSDB()
     ****************************************************************/
    void addSDB(SigDB *new_sdb);

    /****************************************************************
     * removeSDB()
     ****************************************************************/
    void removeSDB(SigDB *sdb);

   /******************************************************************
    * findEdge()
    ******************************************************************/
    int findEdge(Char *sigId, Uint32 time, Uint32 fwd, Uint32 *nextTime);
    void    delSignal(SdbSignal  *sig);
    Uint32  sigIndex(SdbSignal   *sig);

    Uint32  getMaxTime();

    void SigListUpdate();

    void update(void);
    int sigIndex(Char *sigId);
    int isSigSep(Char *sigId);

    inline Int32 get_resolution() { return resolution; }
    const Char *get_resString();
    char *nextSignal();
    void   addSignal(SdbSignal *sig);

    CursorMgr              *cursorMgr;
    Uint32                  okay;

private:

    enum CBReq {
        CBReq_SigData     = 0x0001,
        CBReq_SigList     = 0x0002,
        CBReq_Last
    };

    ClientData              SDBR_SigDataUpdateCbId;
    Char                   *sdbName;
    SigDB                  *sdb;
    Vector<SdbDesc>        *sdbdList;

    Tcl_Interp             *interp;
    Tcl_HashTable           d_sigHash;
    String                  instName;
    Uint32                  sigId;
    Int32                   resolution;

    Uint32                  maxTime;
    Uint32                  maxTimeValid;
    Uint32                  d_cbReqs;

    static void IdleProc(ClientData clientData);

    static int SDBR_SigDataUpdateCB(
            ClientData        clientData,
            Tcl_Interp       *interp,
            int               objc,
            Tcl_Obj          *const objv[]);

    Int32 Configure(Uint32 argc, Char **argv, Uint32 flags);

    int    addSignal(Uint32 argc, Char **argv);

    /****************************************************************
     * appendCutBuffer()
     ****************************************************************/
    int    appendCutBuffer(Uint32 argc, char **argv);

    /****************************************************************
     * putCutBuffer()
     ****************************************************************/
    int    putCutBuffer(Uint32 argc, Char **argv);

    int    addSeparator(Uint32 argc, Char **args);
    int    moveSignal(Uint32 srcIdx, Uint32 destIdx);


    void setupCB(CBReq req);

    enum {
        Opt_SdbName,
        Opt_NumOpts
    };

    /*************************************************************************
     * getConfigSpec()
     *************************************************************************/
    Tk_ConfigSpec *getConfigSpec();
                
    /*************************************************************************
     * optionSpecified()
     *************************************************************************/
    Uint32 optionSpecified(Uint32 idx) {
        return (getConfigSpec()[idx].specFlags & TK_CONFIG_OPTION_SPECIFIED);
    }

    static char d_idBuf[1024];
};


#endif /*INCLUDED_SDB_READER_H*/



