/****************************************************************************
 *                              SDBMgr.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements a manager for multiple SDB's.
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "SdbMgr.h"
#include <vector.h>
#include "ivi_String.h"
#include "ConfigDB.h"
#include <tcl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "WidgetManager.h"
#include "CallbackMgr.h"
#include "TclInterpObj.h"
#include "TclListObj.h"
#include "SigDB.h"
#include "CmdSwitcher.h"

class SDBMgr {
    public:
        SDBMgr(void);

/*        Vector<SDBInfo>        *sdbInfoList; */
        Vector<SigDB>          *sdbInstList;
};

static SDBMgr   *Globl_SDBMgr = NULL;

/**********************************************************
 * SDBMgr()
 **********************************************************/
SDBMgr::SDBMgr(void)
{
    sdbInstList    = new Vector<SigDB>(); 
/*    sdbInfoList    = new Vector<SDBInfo>(); */
}

#ifdef UNDEFINED
/**********************************************************
 * SDBMgr_RegisterSDB()
 **********************************************************/
Int32 SDBMgr_RegisterSDB(SDBInfo *sdbInfo)
{
    SDBMgr *dMgr = Globl_SDBMgr;
    dMgr->sdbInfoList->append(sdbInfo);

    return 0;
}
#endif /* UNDEFINED */

/**********************************************************
 * SDBMgr_NewSDB()
 **********************************************************/
SigDB *SDBMgr_NewSDB(
        const char   *dfioTypeName,
        Tcl_Interp   *interp,
        int           argc,
        char        **argv)
{
    SDBMgr     *dMgr = Globl_SDBMgr;
    SigDB      *sdb  = 0;
    Uint32      i;
  
    if (!dfioTypeName) {
        sdb = new SigDB(interp, argc, argv);
    } 

#ifdef UNDEFINED
    else {
        for (i=0; i<dMgr->dfioList->length(); i++) {
            dInfo = dMgr->dfioList->idx(i);
            if (dInfo->getName().equal(dfioTypeName)) {
                break;
            }
            dInfo = 0;
        }
    }

    if (dInfo) {
        return dInfo->newDFIO();
    } else {
        fprintf(stderr, "ERROR :: dfioFind on \"%s\" failed\n",
                (dfioTypeName)?dfioTypeName:"null");
        if (dMgr->dfioList->length()) {
            dInfo = dMgr->dfioList->idx(0);
            fprintf(stderr, "\tDefaulting to DFIO \"%s\"\n",
                    dInfo->getName().value());
            return dInfo->newDFIO();
        } else {
            fprintf(stderr, "ERROR :: no DFIO's registered\n");
            fprintf(stderr, "\tExiting now...\n");
            exit(1);
        }
    }
#endif

    dMgr->sdbInstList->append(sdb); 

    return sdb;
}

/**********************************************************
 * SDBMgr_RemoveSDB()
 **********************************************************/
void SdbMgr_RemoveSDB(SigDB *sdb)
{
    SDBMgr     *dMgr = Globl_SDBMgr;
    Uint32      i, x;

    dMgr->sdbInstList->remove(sdb);
    CallbackMgr_Invoke(CBTYPE_SDB_MGR_UPDATE, 0, 0, 0);
}

typedef enum {
    SDBM_Create,
    SDBM_List,
    SDBM_NumCmds
};

static CmdSwitchStruct  sdbmgr_cmds[] = {
    {"create",             SDBM_Create },
    {"list",               SDBM_List   },
    {"",                   0           }
};

/********************************************************************
 * SdbMgr_AddSDB()
 ********************************************************************/
void SdbMgr_AddSDB(SigDB *sdb)
{
    SDBMgr *dMgr = Globl_SDBMgr;

    dMgr->sdbInstList->append(sdb); 

    CallbackMgr_Invoke(CBTYPE_SDB_MGR_UPDATE, 0, 0, 0);
}

/********************************************************************
 * SdbMgr_TclCmd()
 *
 * Manages SDB instances...
 ********************************************************************/
static int SdbMgr_TclCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        char            **argv)
{
    SDBMgr        *dMgr = Globl_SDBMgr;
    Char          *typeName = 0;
    Uint32         i, idx = 0;
    TclInterpObj   i_obj(interp);
    SigDB         *sdb =0;
    Int32          cmd;
    TclListObj     list(0);
    Uint32         newArgc=0;
    char          *newArgv[64];

    if (argc < 2) {
        Tcl_AppendResult(interp, "too few args", 0);
        return TCL_ERROR;
    }

    cmd = CmdSwitch(sdbmgr_cmds, argv[1]);

    switch (cmd) {
        case SDBM_Create:
            if (argc < 3) {
                return TCL_ERROR;
            }

            if (WidgetMgr_GetObjHandle(interp, WIDGET_TYPE_SDB, argv[2])) {
                i_obj << (TclListObj(interp) 
                    << "DFIO instance " << argv[1] << " already exists").end();
                return TCL_ERROR;
            }

            newArgv[newArgc++] = "sdb";

            for (idx=2; idx<argc; idx++) {
                if (String::equal(argv[idx], "-type")) {
                    idx++;
                    typeName = argv[idx];
                } else {
                    newArgv[newArgc++] = argv[idx];
                }
            }
            sdb = SDBMgr_NewSDB(typeName, interp, newArgc, newArgv);

            /**** Invoke List-change callback. ****/
            CallbackMgr_Invoke(CBTYPE_SDB_MGR_UPDATE, 0, 0, 0);

            Tcl_SetObjResult(interp, Tcl_NewStringObj(sdb->getInstName(), -1));
            break;

        case SDBM_List:
            list(interp);
            for (i=0; i<dMgr->sdbInstList->length(); i++) {
                sdb = dMgr->sdbInstList->idx(i);
                list << sdb->getInstName();
            }

            Tcl_SetObjResult(interp, list.end());
            break;

        default:
            fprintf(stderr, "SdbMgr ERROR :: Unknown cmd \"%s\"\n", argv[1]);
            return TCL_ERROR;
    }

    return TCL_OK;
}

static const char *prv_SdbCb[] = {
    CBTYPE_SDB_SIGDATA_UPDATE,
    CBTYPE_SDB_MGR_UPDATE,
    ""
};

/**********************************************************
 * SDBMgr_Init()
 **********************************************************/
extern "C" int SDBMgr_Init(Tcl_Interp *interp)
{
    Uint32 i = 0;

    if (!Globl_SDBMgr) {
        Globl_SDBMgr = new SDBMgr();
    }

    Tcl_CreateCommand(interp, "sdb_mgr", 
            (Tcl_CmdProc *)SdbMgr_TclCmd, 0, 0);
    WidgetMgr_AddType(interp, WIDGET_TYPE_SDB);

    while (prv_SdbCb[i][0]) {
        CallbackMgr_AddCbType(prv_SdbCb[i]);
        i++;
    }

    return TCL_OK;
}


