/****************************************************************************
 *                       RemoteAppClientConnection.cc
 *
 * Author: Matthew Ballance
 * Desc: 
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "RemoteAppClientConnection.h"
#include "RemoteAppConnectionListener.h"
#include "IviRemoteAppIdx.h"
#include <sys/types.h>

#ifndef __MINGW32__
#include <sys/socket.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <arpa/inet.h>
#endif /* __MINGW32__ */
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

class IntChClientListener : public RemoteAppConnectionListener {
    public:
        IntChClientListener(RemoteAppClientConnection *clnt) :
            RemoteAppConnectionListener(IVI_CMD_CH_IDX_INTERNAL)
        {
            d_clnt = clnt;
        }

        virtual void Receive(Uint32 idx, Uint32 len, Uchar *data)
        {
            Uint32 cmd;
            Uint32 bidx=0;

            cmd = RemoteAppConnection::ReadUint32(data, bidx);

            switch (cmd) {
                case IVI_INT_CMD_CONNECT_ACK:
                    d_clnt->HandleConnectComplete();
                    break;

                default:
                    fprintf(stderr, "INTERNAL ERROR: IntChClientListener "
                            "received unexpected command \"%d\"\n", cmd);
                    break;
            }
        }

    private:
        RemoteAppClientConnection   *d_clnt;
};

/********************************************************************
 * RemoteAppClientConnection()
 ********************************************************************/
RemoteAppClientConnection::RemoteAppClientConnection(
        Tcl_Interp *interp, Int32 port)
{
    /*** Add the internal channel-cmd listener for this channel ****/
    AddListener(new IntChClientListener(this));

    d_connected       = 0;
    d_connectComplete = 0;

    d_channel = Tcl_OpenTcpClient(interp, port, "localhost", 0, 0, 0);

    if (!d_channel) {
        fprintf(stderr, "ERROR: cannot create tcp client: %s\n",
            Tcl_GetStringResult(interp));
    } 

    if (Tcl_SetChannelOption(interp, d_channel, "-blocking", 
        "false") != TCL_OK) {
        fprintf(stderr, "-blocking failed: %s\n", Tcl_GetStringResult(interp));
        fflush(stderr);
    }

    if (Tcl_SetChannelOption(interp, d_channel, "-buffering", "none") != TCL_OK)
    {
        fprintf(stderr, "-blocking failed: %s\n", Tcl_GetStringResult(interp));
        fflush(stderr);
    }

    if (Tcl_SetChannelOption(interp, d_channel, "-encoding", 
            "binary") != TCL_OK) {
        fprintf(stderr, "-encoding failed: %s\n", Tcl_GetStringResult(interp));
        fflush(stderr);
    }

    if (Tcl_SetChannelOption(interp, d_channel, "-translation", 
            "binary") != TCL_OK) {
        fprintf(stderr, "-translation failed: %s\n", Tcl_GetStringResult(interp));
        fflush(stderr);
    }

    d_port       = port;
    d_backoffIdx = 0;

#if 0
#ifndef __MINGW32__
    int ret;
    struct sockaddr_in   addr;

    /*** Add the internal channel-cmd listener for this channel ****/
    AddListener(new IntChClientListener(this));

    d_connected       = 0;
    d_connectComplete = 0;

    d_skt = socket(PF_INET, SOCK_STREAM, 0);
    fcntl(d_skt, F_SETFL, O_NONBLOCK);

    d_port       = port;
    d_backoffIdx = 0;


    memset(&addr, 0, sizeof(struct sockaddr_in));
    addr.sin_family        = AF_INET;
    addr.sin_port          = htons(d_port);
    addr.sin_addr.s_addr   = htonl(INADDR_LOOPBACK);

    ret = connect(d_skt, (const struct sockaddr *)&addr,
            sizeof(addr));

    if (ret < 0) {
        if (errno == EINPROGRESS) {
            SetupRetry();
        }
    } else {
        HandleConnect();
    }
#else
    fprintf(stderr, "TODO: RemoteAppClientConnection()\n");
    fflush(stderr);
#endif
#endif
}

/********************************************************************
 * ~RemoteAppClientConnection()
 ********************************************************************/
RemoteAppClientConnection::~RemoteAppClientConnection()
{
    Tcl_DeleteChannelHandler(d_channel, 
            &RemoteAppConnection::EventProc, this);
}

/********************************************************************
 * HandleDisconnect()
 ********************************************************************/
void RemoteAppClientConnection::HandleDisconnect()
{
    Tcl_DeleteChannelHandler(d_channel, 
            &RemoteAppConnection::EventProc, this);
}

/********************************************************************
 * Connect()
 *
 * Connect to the master process. 
 * - Send a This requires us to poll while 
 ********************************************************************/
int RemoteAppClientConnection::Connect()
{
    Uchar   buf[8];
    Uint32  idx=0, pid;

    Tcl_CreateChannelHandler(d_channel, TCL_READABLE, 
        &RemoteAppConnection::EventProc, this);

    /**** Now, send a message to the master indicating which process
     **** we're in...
     ****/
#ifndef __MINGW32__
    pid = getpid();
#else
    char *pid_var = getenv("IVI_PROC_ID");

    if (!pid_var) {
        fprintf(stderr, "ERROR: IVI_PROC_ID not set\n");
        fflush(stderr);
    } else {
        pid = strtol(pid_var, 0, 10);
    }
#endif

    WriteUint32(IVI_INT_CMD_CONNECT, buf, idx);
    WriteUint32(pid, buf, idx);

    Send(IVI_CMD_CH_IDX_INTERNAL, idx, (Uchar *)buf);


    /**** Now, wait for an Ack that the master is ready ****/
    while (!d_connectComplete) {
        Poll(REMOTE_POLL_FOREVER);
    }

    if (d_connectComplete < 0) {
        return d_connectComplete;
    }

    return 0;
}

/********************************************************************
 * TryConnect()
 ********************************************************************/
Int32 RemoteAppClientConnection::TryConnect()
{
#ifndef __MINGW32__
    int                   ret;
    struct sockaddr_in    addr;

    memset(&addr, 0, sizeof(struct sockaddr_in));
    addr.sin_family        = AF_INET;
    addr.sin_port          = htons(d_port);
    addr.sin_addr.s_addr   = htonl(INADDR_LOOPBACK);

    ret = connect(d_skt, (const struct sockaddr *)&addr, sizeof(addr));

    /**** If the connection has already been made, then don't 
     **** return an error...
     ****/
    if (ret < 0 && (errno == EISCONN)) {
        ret = 0;
    }
   
    return ret;
#else
    fprintf(stderr, "TODO: RemoteAppClientConnection::TryConnect()\n");
    fflush(stderr);
#endif
}

/********************************************************************
 * SetupRetry()
 ********************************************************************/
void RemoteAppClientConnection::SetupRetry()
{
    if (d_backoffIdx < d_backoffLen) {
        d_timerToken = Tcl_CreateTimerHandler(d_backoffPattern[d_backoffIdx],
                &RemoteAppClientConnection::ConnectProc, this);
        d_backoffIdx++;
    } 
}

/********************************************************************
 * ConnectProc()
 ********************************************************************/
void RemoteAppClientConnection::ConnectProc(ClientData clientData)
{
    int  ret;
    RemoteAppClientConnection *client = 
        (RemoteAppClientConnection *)clientData;

    ret = client->TryConnect();

    if (ret == 0) {
        client->HandleConnect();
    } else if (ret < 0) {
        fprintf(stderr, "TryConnect() ERROR\n");
        fflush(stderr);
        client->SetupRetry();
        Tcl_DeleteChannelHandler(client->d_channel, 
                &RemoteAppConnection::EventProc, client);
    }
}

/********************************************************************
 * HandleConnect()
 ********************************************************************/
void RemoteAppClientConnection::HandleConnect()
{
    d_connected = true;
}

/********************************************************************
 * HandleConnectComplete()
 ********************************************************************/
void RemoteAppClientConnection::HandleConnectComplete()
{
    d_connectComplete = 1;
}

/********************************************************************
 * RemoteAppClientConnection_InstCmd()
 ********************************************************************/
static int RemoteAppClientConnection_InstCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        const char      **argv)
{
    RemoteAppClientConnection *conn;
    Int32 port;

    if (argc != 2) {
        return TCL_ERROR;
    }

    port = strtoul(argv[1], 0, 0);

    conn = new RemoteAppClientConnection(interp, port);

    return TCL_OK;
}

/********************************************************************
 * RemoteAppClientConnection_Init()
 ********************************************************************/
extern "C" int RemoteAppClientConnection_Init(Tcl_Interp *interp)
{
    Tcl_CreateCommand(interp, "remote_app_client",
            RemoteAppClientConnection_InstCmd, 0, 0);

    return TCL_OK;
}


Int32 RemoteAppClientConnection::d_backoffPattern[] = {
    1, 2, 2, 4, 4, 8, 8, 16, 16, 32, 32, 64, 128, 256, 512, 1024, 2048, 4096,
    1, 2, 2, 4, 4, 8, 8, 16, 16, 32, 32, 64, 128, 256, 512, 1024, 2048, 4096,
    1, 2, 2, 4, 4, 8, 8, 16, 16, 32, 32, 64, 128, 256, 512, 1024, 2048, 4096,
    1, 2, 2, 4, 4, 8, 8, 16, 16, 32, 32, 64, 128, 256, 512, 1024, 2048, 4096};
Int32 RemoteAppClientConnection::d_backoffLen = 
    sizeof(RemoteAppClientConnection::d_backoffPattern)/sizeof(Int32);

