/****************************************************************************
 *                               PixmapObj.h
 *
 * Author: Matthew Ballance
 * Desc:   Implements a wrapper around a pixmap... This allows us to 
 *         implement various back-ends and use the same drawing code...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#ifndef INCLUDED_PIXMAP_OBJ_H
#define INCLUDED_PIXMAP_OBJ_H
#include <tk.h>
#include "vector.h"
#include "types.h"
#include "GCObj.h"

class GCObj;
class PixmapPt;
class PixmapRange;
class PixmapObj {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * PixmapObj()
         ************************************************************/
        PixmapObj();

        /************************************************************
         * ~PixmapObj()
         ************************************************************/
        virtual ~PixmapObj();

        /************************************************************
         * line()
         ************************************************************/
        virtual void line(GC gc, Uint32 x1, Uint32 y1, Uint32 x2, Uint32 y2);

        /************************************************************
         * rect()
         ************************************************************/
        virtual void rect(GC gc, Uint32 x, Uint32 y, 
                Uint32 width, Uint32 height);

        /************************************************************
         * fill_rect()
         ************************************************************/
        virtual void fill_rect(GC gc, Uint32 x, Uint32 y, 
                Uint32 width, Uint32 height);

        /************************************************************
         * ComputeTextLayout()
         ************************************************************/
        virtual Tk_TextLayout ComputeTextLayout(
                Tk_Font font, Char *str, Int32 len, Int32 wraplen,
                Tk_Justify justify,
                Int32 flags, Int32 *width, Int32 *height);

        /************************************************************
         * DrawTextLayout()
         ************************************************************/
        virtual void DrawTextLayout(
                GC gc, Tk_TextLayout layout, Uint32 x, Uint32 y, 
                Int32 first, Int32 last);

        /************************************************************
         * FreeTextLayout()
         ************************************************************/
        virtual void FreeTextLayout(Tk_TextLayout layout);

        /************************************************************
         * createGC()
         ************************************************************/
        virtual GC createGC(GCObj &gcObj);

        /************************************************************
         * freeGC()
         ************************************************************/
        virtual void freeGC(GCObj &gcObj);

        /************************************************************
         * copy 
         *
         * Copy data from (x,y)-(x+width,y+height) to (x1, y1)
         ************************************************************/
        virtual void copy(PixmapObj *obj, Uint32 x, Uint32 y, 
                Uint32 width, Uint32 height, Uint32 x1, Uint32 y1);

        /************************************************************
         * copy 
         *
         * Copy data from (x,y)-(x+width,y+height) to (x1, y1)
         ************************************************************/
        virtual void copy(PixmapObj *obj, Uint32 x, Uint32 y, 
                Uint32 width, Uint32 height);

        /************************************************************
         * operator ()
         ************************************************************/
        virtual PixmapPt operator () (Uint32 x, Uint32 y);

        /************************************************************
         * operator ()
         ************************************************************/
        virtual PixmapRange operator () (
                Uint32 x, Uint32 y, Uint32 width, Uint32 height);

        /************************************************************
         * operator ()
         ************************************************************/
        virtual PixmapRange operator () ();

        /************************************************************
         * operator =
         ************************************************************/
        virtual void operator = (const PixmapRange &rng);

        /************************************************************
         * blank()
         *
         * sets the color of the pixmap to the background color...
         ************************************************************/
        virtual void blank();

        /************************************************************
         * setBgColor()
         ************************************************************/
        virtual void setBgColor(Tk_3DBorder background) { 
            d_background = background; 
        }

        /************************************************************
         * getBgColor()
         ************************************************************/
        virtual Tk_3DBorder getBgBorderColor() { return d_background; }

        /************************************************************
         * width(Uint32)
         ************************************************************/
        virtual void width(Uint32 width) { width_height(width, d_height);   }

        /************************************************************
         * height(Uint32)
         ************************************************************/
        virtual void height(Uint32 height) { width_height(d_width, height); }

        /************************************************************
         * width_height()
         ************************************************************/
        virtual void width_height(Uint32 width, Uint32 height);

        /************************************************************
         * width()
         ************************************************************/
        virtual Uint32 width() const { return d_width; }

        /************************************************************
         * height()
         ************************************************************/
        virtual Uint32 height() const { return d_height; }

        /************************************************************
         * Pixmap types
         *
         * - XPixmap
         *   Pixmap for drawing on X11 objects
         *
         * - PsPixmap
         *   Pixmap for creating postscript output...
         ************************************************************/
        enum PixmapType {
            PixmapNone = 1,
            XPixmap,
            PsPixmap,
            NumPixmapTypes
        };

        /************************************************************
         * getType()
         *
         * returns the type of this pixmap...
         ************************************************************/
        virtual PixmapObj::PixmapType getType() const;

        /************************************************************
         * addGC()
         ************************************************************/
        virtual void addGC(GCObj &gc);

        /************************************************************
         * removeGC()
         ************************************************************/
        virtual void removeGC(GCObj &gc);

        /************************************************************
         * addGCs()
         ************************************************************/
        virtual void addGCs(GCObj *objs, Uint32 num_objs);

        /************************************************************
         * removeGCs()
         ************************************************************/
        virtual void removeGCs(GCObj *objs, Uint32 num_objs);

    /****************************************************************
     * Public Data
     ****************************************************************/
    public:

    /****************************************************************
     * Protected Data
     ****************************************************************/
    protected:
        Uint32        d_height;
        Uint32        d_width;
        Tk_3DBorder   d_background;

    /****************************************************************
     * Private Methods
     ****************************************************************/
    private:

    /****************************************************************
     * Private Data
     ****************************************************************/
    private:
        Vector<GCObj> d_gcObjs;
};

/************************************************************
 * PixmapPt
 *
 * Represents a dest point on a pixmap...
 ************************************************************/
class PixmapPt {
    public:
        PixmapPt(PixmapObj *pixmap, Uint32 x, Uint32 y) :
                    d_pixmap(pixmap), d_x(x), d_y(y) { ; }

        virtual void operator = (const PixmapRange &rng);

        Uint32            d_x;
        Uint32            d_y;
        PixmapObj        *d_pixmap;
};

/************************************************************
 * PixmapRange
 *
 * Represents a pixmap source location (pt and width/height)
 ************************************************************/
class PixmapRange {
    public:
        PixmapRange(PixmapObj *pixmap, Uint32 x, Uint32 y, 
            Uint32 width, Uint32 height) :
        d_pixmap(pixmap), d_x(x), d_y(y), d_width(width),
        d_height(height) { ; }

        PixmapObj    *d_pixmap;
        Uint32        d_x, d_y;
        Uint32        d_width;
        Uint32        d_height;
};

#endif /* INCLUDED_PIXMAP_OBJ_H */

