/****************************************************************************
 *                           IviUintHashTab.h
 *
 * Author: Matthew Ballance
 * Desc:   Implements an efficient sparse lookup table for integer keys
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#ifndef INCLUDED_IVI_UINT_HASH_TAB_H
#define INCLUDED_IVI_UINT_HASH_TAB_H
#include "types.h"
#include "IviUintHashTabEntry.h"
#include "vector.h"

class IviUintHashTabBase {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * IviUintHashTabBase()
         ************************************************************/
        IviUintHashTabBase();

        /************************************************************
         * ~IviUintHashTabBase()
         ************************************************************/
        virtual ~IviUintHashTabBase();

        /************************************************************
         * base_add()
         ************************************************************/
        void base_add(IviUintHashTabEntryBase *entry);

        /************************************************************
         * add()
         ************************************************************/
        virtual void add(IviUintHashTabEntryBase *e) { base_add(e); }

        /************************************************************
         * base_remove()
         ************************************************************/
        void base_remove(IviUintHashTabEntryBase *e);

        /************************************************************
         * remove()
         ************************************************************/
        virtual void remove(IviUintHashTabEntryBase *e) { base_remove(e); }

        /************************************************************
         * base_find()
         ************************************************************/
        IviUintHashTabEntryBase *base_find(Uint32 key);

        /************************************************************
         * find()
         ************************************************************/
        virtual IviUintHashTabEntryBase *find(Uint32 key) {
            return base_find(key);
        }

    /****************************************************************
     * Private Data
     ****************************************************************/
    protected:
        IviUintHashTabLevel           *d_treeRoot;
        Uint32                         d_numEntries;
};


/********************************************************************
 * IviUintHashTab
 ********************************************************************/
template <class eType> class IviUintHashTab : public IviUintHashTabBase {

    /****************************************************************
     * Public Methods
     ****************************************************************/
    public:

        /************************************************************
         * IviUintHashTab()
         ************************************************************/
        IviUintHashTab() { ; }

        /************************************************************
         * ~IviUintHashTab()
         ************************************************************/
        virtual ~IviUintHashTab() { ; } 

        /************************************************************
         * find()
         ************************************************************/
        IviUintHashTabEntry<eType> *find(Uint32 key);

        /************************************************************
         * getList()
         ************************************************************/
        void getList(Vector< IviUintHashTabEntry<eType> > &result);

        /************************************************************
         * getListHelper()
         ************************************************************/
        void getListHelper(
                Vector< IviUintHashTabEntry<eType> > &result,
                IviUintHashTabLevel                *level);

};

/********************************************************************
 * find()
 ********************************************************************/
template <class eType> 
    IviUintHashTabEntry<eType> *IviUintHashTab<eType>::find(Uint32 key)
{
    return (IviUintHashTabEntry<eType> *)base_find(key);
}

/********************************************************************
 * getListHelper()
 ********************************************************************/
template <class eType> void IviUintHashTab<eType>::getListHelper(
        Vector< IviUintHashTabEntry<eType> >        &result,
        IviUintHashTabLevel                       *level)
{
    if (level->getDepth() == 7) {
        for (Uint32 i=0; i<16; i++) {
            if (level->d_children[i]) {
                result.append(
                    (IviUintHashTabEntry<eType> *)level->d_children[i]);
            }
        }
    } else {
        for (Uint32 i=0; i<16; i++) {
            if (level->d_children[i]) {
                getListHelper(result, 
                        (IviUintHashTabLevel *)level->d_children[i]);
            }
        }
    }
}

/********************************************************************
 * getList()
 ********************************************************************/
template <class eType> void IviUintHashTab<eType>::getList(
        Vector< IviUintHashTabEntry<eType> >        &result)
{
    IviUintHashTabLevel   *level = d_treeRoot;

    /**** scan through the hash in order... ****/
    for (Uint32 i=0; i<16; i++) {
        if (level->d_children[i]) {
            getListHelper(result, 
                    (IviUintHashTabLevel *)level->d_children[i]);
        }
    }
}

#endif /* INCLUDED_IVI_UINT_HASH_TAB_H */


