/*****************************************************************************
 *                             CallbackMgr.h 
 *
 * Author: Matthew Ballance
 * Desc:   Implements a callback interface for C and TCL.
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *****************************************************************************/
#ifndef INCLUDED_CALLBACK_MGR_H
#define INCLUDED_CALLBACK_MGR_H
#include "types.h"
#include <tcl.h>
#include "CallbackTypes.h"
#include "CallbackList.h"

#ifdef __cplusplus
extern "C" {
#endif

/************************************************************************
 * CallbackMgr_Init()
 ************************************************************************/
int CallbackMgr_Init(Tcl_Interp *interp);

/************************************************************************
 * CallbackMgr_AddCbType()
 *
 * Adds a callback-type to the manager. This must be done before 
 * callbacks may be added on this type
 *
 * Parameters:
 * - cbTypeName
 *   Type-name for the callback...
 ************************************************************************/
int CallbackMgr_AddCbType(const char *cbTypeName);


/************************************************************************
 * CallbackMgr_AddCb()
 *
 * Parameters:
 * - cbTypeName
 *   Callback type
 *
 * - cbInstName
 *   Instance of this callback-type. May be NULL for instance-less 
 *   callbacks
 *
 * - cbFunc
 *   Callback function to be called 
 *
 * - cbUserData
 *   user-data passed into the callback when it is triggered
 *
 * - cbId
 *   handle identifying this callback... This handle is used to delete
 *   a callback when the owner is deleted
 ************************************************************************/
int CallbackMgr_AddCb(
        const char      *cbTypeName,
        const char      *cbInstName,
        Tcl_ObjCmdProc   cbFunc,
        ClientData       cbUserData,
        ClientData      *cbId);

/************************************************************************
 * CallbackMgr_DeleteCb()
 ************************************************************************/
int CallbackMgr_DeleteCb(ClientData cbId);

/************************************************************************
 * CallbackMgr_Invoke()
 *
 * Parameters:
 * - cbTypeName
 *   String representing the callback type
 *
 * - cbInstName
 *   Instance of this callback to invoke. May be NULL for instance-less
 *   callbacks.
 ************************************************************************/
int CallbackMgr_Invoke(
        const char  *cbTypeName,
        const char  *cbInstName,
        Uint32       objc,
        Tcl_Obj     *const objv[]);

/************************************************************************
 * CallbackMgr_Disable()
 ************************************************************************/
int CallbackMgr_Disable(
        const char *cbTypeName,
        const char *cbInstName);

/************************************************************************
 * CallbackMgr_Enable()
 ************************************************************************/
int CallbackMgr_Enable(
        const char *cbTypeName,
        const char *cbInstName);

#ifdef __cplusplus
};
#endif

/****************************************************************************
 *                   C++ CallbackMgr object definitions
 ****************************************************************************/
#ifdef __cplusplus

#include "ivi_String.h"
#include "vector.h"

class CallbackList;
class CallbackMgr {

    public:
        CallbackMgr(Tcl_Interp *interp);

        int AddCbType(const char *cbTypeName);
        int GetCbTypes(const char *cbTypeSpec, Vector<String> **result);
        int AddCb(const char *cbTypeName, const char *cbInstName, 
                Tcl_ObjCmdProc cbFunc,
                ClientData cbUserData, ClientData *cbId);

        static CallbackList *GetCbList(
                const char *cbTypeName, const char *cbInstName);

        /*
        Char         *GetCbId(CallbackInfo *cb);
        CallbackInfo *LookupCb(Char *cbId);
         */

        int Invoke(const char *cbTypeName, const char *cbInstName, 
                Uint32 argc, Tcl_Obj *const objv[]);

        int Disable(const char *cbTypeName, const char *cbInstName);
        int Enable(const char *cbTypeName, const char *cbInstName);

        inline Tcl_Interp *get_interp(void) { return interp; }
        inline Tcl_Interp *getInterp(void) { return interp; }

        static CallbackMgr *GetCbMgr();
        static void Init(Tcl_Interp *interp);

    private:
        Tcl_HashTable            cbTypeHash;
        Tcl_HashTable            cbHashTab;
        Tcl_Interp              *interp;
        Vector<String>          *retVect;
        Vector<String>          *strVect;
        Uint32                   numCBs;
        static CallbackMgr      *Globl_CbMgr;
};
#endif /* __cplusplus */

#endif /* INCLUDED_CALLBACK_MGR_H */

