/*****************************************************************************
 *                               BitVector.h
 *
 * Author: Matthew Ballance
 * Desc:   Implements a bit-vector and arithmetic
 *
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 *
 *****************************************************************************/
#ifndef INCLUDED_BIT_VECTOR_H
#define INCLUDED_BIT_VECTOR_H
#include "types.h"
#include "ivi_String.h"
#include "vpi_user.h"
#include "DFIOValChg.h"

class BitVector {

    public:
        typedef Uint32      RadixType;

        static const Uint32 Radix_Mask     = 0x0FF;
        static const Uint32 Radix_FlagMask = 0xF00;
        enum RadixEnum {
            Radix_Bin         = 0x001,
            Radix_Hex         = 0x002,
            Radix_Dec         = 0x003,
            Radix_Oct         = 0x004,
            Radix_Str         = 0x005,
            Radix_Signed      = 0x100,
            Radix_Sized       = 0x200,
            Radix_Typed       = 0x400
        };

        enum FmtFlags {
            Fmt_HasXZ = 0x0001,
            Fmt_ExclZ = 0x0002,
            Fmt_NumFlg
        };

        BitVector(void);
        BitVector(const String      &bitstr);
        BitVector(const String      &bitstr, Uint32 flags);
        BitVector(const BitVector   &bv);

        const BitVector operator ~ (void);
        const BitVector operator + (const BitVector &);
        const BitVector operator - (const BitVector &);
        const BitVector operator & (const BitVector &);
        const BitVector operator | (const BitVector &);
        const BitVector operator ^ (const BitVector &);
        const BitVector operator * (const BitVector &);
        Uint32 operator == (const BitVector &);
        void operator () (const String &bitstr);
        void operator = (const BitVector &);


        void setLen(Int32 len) { 
            storage.setLen(len);
            display.setLen(len);
        }

        String toString(void);
        static void toString(
                    String       &in,
                    String       &out,
                    Uint32 disp_radix);

        static Int32 binToBinV(
                const String    &in,
                Uint32           start_idx,
                String          &out,
                Uint32           size,
                Uint32           flags);

        static Int32 decToBinV(
                const String    &in,
                Uint32           start_idx,
                String          &out,
                Uint32           size,
                Uint32           flags);

        static Int32 hexToBinV(
                const String    &in,
                Uint32           start_idx,
                String          &out,
                Uint32           size,
                Uint32           flags);

        static Int32 toBin(
                const String &in,
                String       &out,
                Uint32       *flags);


        inline Uint32 width(void) const {return d_width;}
        void setRadix(Uint32 new_radix);
        void setWidth(Uint32 new_width);
        inline Uint32 getRadix(void) { return radix; }

        inline char *rawValue() { return storage.value(); }

        void get_vecval(Uint32 width, s_vpi_vecval **vecval);

        static void get_vecval(
                String        &str,
                Uint32         width, 
                Uint32         radix_f,
                s_vpi_vecval **vecval);

        static void get_vecval(
                const DFIOVal *str,
                Uint32         width, 
                Uint32         radix_f,
                s_vpi_vecval **vecval);

        static void fmtDec(
                const String   &in, 
                String         &out,
                Uint32          flags);

        static void fmtDec(
                const DFIOVal  *in,
                Uint32          in_len,
                String         &out,
                Uint32          flags,
                Uint32         &infoFlags);

        static void fmtDec(
                const String   &in, 
                String         &out,
                Uint32          flags,
                Uint32         &infoFlags);

        static void fmtHex( 
                const String       &in,
                      String       &out,
                      Uint32        disp_radix);

        static void fmtHex( 
                const String       &in,
                      String       &out,
                      Uint32        disp_radix,
                      Uint32       &flags);

        static void fmtHex(
                const DFIOVal  *in,
                Uint32          in_len,
                String         &out,
                Uint32          flags,
                Uint32         &infoFlags);

        static void fmtString( 
                const String       &in,
                      String       &out,
                      Uint32        disp_radix);

        static void fmtString(
                const DFIOVal  *in,
                Uint32          in_len,
                String         &out,
                Uint32          flags,
                Uint32         &infoFlags);

        static void fmtString( 
                const String       &in,
                      String       &out,
                      Uint32        disp_radix,
                      Uint32       &infoFlags);

        static void fmtBin( 
                const String       &in,
                      String       &out,
                      Uint32        disp_radix);

        static void fmtBin(
                const DFIOVal  *in,
                Uint32          in_len,
                String         &out,
                Uint32          flags,
                Uint32         &infoFlags);

        static void fmtBin( 
                const String       &in,
                      String       &out,
                      Uint32        disp_radix,
                      Uint32       &infoFlags);

        static void fmtOct(
                const String     &in,
                String           &out,
                Uint32            disp_radix);

        static void fmtOct(
                const DFIOVal    *in,
                Uint32            in_len,
                String           &out,
                Uint32            disp_radix,
                Uint32           &infoFlags);

        static void fmtOct(
                const String     &in,
                String           &out,
                Uint32            disp_radix,
                Uint32           &infoFlags);

    private:
        static inline Uint32 isZ(Char ch) { 
            return (ch == 'z' || ch == 'Z')?1:0; }
        static inline Uint32 isX(Char ch) { 
            return (ch == 'x' || ch == 'X')?1:0; }
        static inline Uint32 is1(Char ch) { 
            return (ch == '1')?1:0; }
        static inline Uint32 is0(Char ch) { 
            return (ch == '0')?1:0; }

        static char get_signext(char ch, Uint32 radix_f);

        static void shift_in(
                Uint32       *valv, 
                Uint32        vlen,
                Uint32        val);

        static Uint32 write_digits(
                Uint32         v,
                String        &buf,
                Uint32        &idx,
                Uint32         zero_suppress);

        /************************************************************
         * put_vecval()
         ************************************************************/
        static void put_vecval(s_vpi_vecval *vecval, Uint32 idx, Char val) {
            vecval[idx/32].aval |= 
                (((val=='1')?1:0) << (idx%32));
            vecval[idx/32].bval |= 
                (((val=='z'||val=='Z'||val=='x'||val=='X')?1:0) << (idx%32));
        }

        static Uint32       *tmp_arr;
        static Uint32        tmp_arr_len;
        Uint32      displayValid;
        Uint32      radix;

        Uint32      isSigned;
        Uint32      isSized;

        String      storage;
        String      display;
        Uint32      d_width;

        static s_vpi_vecval *d_vecval;
        static Uint32        d_vecval_len;

        static const Uint32 BASE    = 10000;
        static const Uint32 BBITS   = 16;
        static const Uint32 BDIGITS = 4;
};



#endif /* INCLUDED_BIT_VECTOR_H */


